// -*- mode: java; indent-tabs-mode: t; c-basic-offset: 4; tab-width: 4 -*-

package antichess;

/**
 * <p>A <tt>Piece</tt> represents a piece on a board.</p>
 * <p><tt>Piece</tt>s are immutable</p>
 *
 * @specfield player : Player				// The Player who owns this Piece
 * @specfield type : PieceType				// This Piece's type
 * @specfield position 						// This Piece's position on the board
 **/
public class Piece
{
	private final Player player;
	private final PieceType type;
	private final int row;
	private final int column;

	/**
	 * @effects Constructs a Piece with the given parameters
	 * @requires player != null && type != null
	 */
	public Piece(Player player, PieceType type, int row, int column) {
		this.player = player;
		this.type = type;
		this.row = row;
		this.column = column;
	}
	
	/**
	 * @return this.type
	 **/
	public PieceType getType() {
		return type;
	}

	/**
	 * Convenience method
	 * @return this.type.movement
	 **/
	public MoveGenerator getMovementType() {
		return getType().getMovement();
	}

	/**
	 * @return this.position.row
	 **/
	public int getRow() {
		return row;
	}

	/**
	 * @return this.position.column
	 **/
	public int getColumn() {
		return column;
	}


	/**
	 * @return this.player
	 **/
	public Player getPlayer() {
		return player;
	}


	/**
	 * @return a copy of this Piece, relocated to the specified position.
	 **/
	public Piece moveTo(int row, int column) {
		return new Piece(player, type, row, column);
	}

	/**
	 * @return true if this piece has the same type, player, and
	 * position as the given piece.
	 **/
	public boolean equals(Object o) {
		if(o == null) return false;
		if(!(o instanceof Piece)) return false;
		Piece p = (Piece)o;
		return p.player == player && p.type.equals(type) && p.row == row && p.column == column;
	}

	/**
	 * @return a textual description of this piece
	 **/
	public String toString() {
		StringBuilder sb = new StringBuilder();
		sb.append("Piece[");
		sb.append(player);
		sb.append(" ");
		sb.append(type.getName());
		sb.append("@");
		sb.append(row);
		sb.append(",");
		sb.append(column);
		sb.append("]");
		return sb.toString();
	}
}
