package antichess;

import java.util.ArrayList;
import java.util.List;

/**
 * A GameDescriptor is an immutable ADT that stores the information required
 * for generating a new game.
 * 
 * Fields store by the GameDescriptor:
 * @specfield gameTypeName     		//String which describes the type of game played
 * @specfield players				//List<Player> which lists all the Players playing the game
 * @specfield playerDescriptions	//List<String> which describes the type of player for each
 * @specfield playerTimes			//long[] which contains the total amount of playing time for each player
 * 									//0 means the player is not timed.
 * @specfield allowSaveLoad			//boolean as to whether saving and loading are allowed
 * 
 * 
 * Acceptable values for the fields (space delineated)
 * For gameTypeName 		: Chess Antichess
 * For players				: WHITE BLACK
 * For playerDescription	: Human AI
 * For playerTimes			: all values must be >= 0
 * For allowSaveLoad		: true false
 *
 * Attempting to create a GameDescriptor with something other than these values
 * will cause it to throw an IllegalArgumentException
 * 
 * Values indexed by the same number in players, playerDescriptions, and playerTimes  
 * describe the same player.  All of these sequences must be the same size.
 *
 * The times stored in playerTimes represent the amount on the timer in milliseconds.
 *
 */


public class GameDescriptor {
	
	
	//Fields
	private final String gameTypeName;
	private final String[] acceptableGameTypeNames = { "Chess", "Antichess" };
	
	private final List<Player> players;
	private final Player[] acceptablePlayers = { Player.WHITE, Player.BLACK };
	
	private final List<String> playerDescriptions;
	private final String[] acceptablePlayerDescriptions = {"Human", "AI"};
	
	private final long[] playerTimes;
	
	private final boolean allowSaveLoad;
	
	//AbstractionFunction
	//Map the name to the name
	
	//Representation Invariants
	//Encompassed by the actual specs
	//fields beginning in 'acceptable' store the acceptable values from the spec
	
	/**
	 * Constructor 
	 * @requires all parameters to be not null
	 * @modifies all the spec fields
	 * @effects sets the specfield to the argument of the same name
	 * @throws IllegalArgumentException if one of the fields does not contain
	 * 	a valid value (For valid values see description of the class)
	 */
	public GameDescriptor(String gameTypeName,
			List<Player> players, List<String> playerDescriptions,
			long[] playerTimes, boolean allowSaveLoad){
		this.gameTypeName = gameTypeName;
		this.players = players;
		this.playerDescriptions = playerDescriptions;
		this.playerTimes = playerTimes;
		this.allowSaveLoad = allowSaveLoad;
		checkFields();
		
	}
	
	
	private void checkFields(){
		if (gameTypeName == null){
			throw new IllegalArgumentException("gameTypeName is null");
		}if (players == null){
			throw new IllegalArgumentException("players is null");
		}if (playerDescriptions == null){
			throw new IllegalArgumentException("playerDescriptions is null");
		}if (playerTimes == null){
			throw new IllegalArgumentException("playerTimes is null");
		}
		
		
		if ((players.size()!=playerDescriptions.size())||
		(players.size()!=playerTimes.length)){
			throw new IllegalArgumentException("Player lists are not the same length");
		}
		
		
		//temp variable used to check things
		boolean violatesRules;
		
		//checks for gameTypeName
		violatesRules = true;
		for (String an : acceptableGameTypeNames){
			if (an.equals(gameTypeName)){
				violatesRules = false;
			}
		}if(violatesRules){
			throw new IllegalArgumentException("Invalid gameTypeName "+gameTypeName);
		}
		
		//checks for players
		for (Player pl: players){
			violatesRules = true;
			for (Player pla: acceptablePlayers){
				if (pl.equals(pla))violatesRules = false;
			}if(violatesRules){
				throw new IllegalArgumentException("Invalid player "+pl.toString());
			}
		}
		
		//checks for playerDescriptions
		for (String pl: playerDescriptions){
			violatesRules = true;
			for (String pla: acceptablePlayerDescriptions){
				if (pl.equals(pla))violatesRules = false;
			}if(violatesRules){
				throw new IllegalArgumentException("Invalid playerDescription "+pl);
			}
		}
		
		//checks for playerTimes
		for (long time : playerTimes ){
			if (time<0){
				throw new IllegalArgumentException("Time in playerTimes is negative");
			}
		}
		
		
	}
	
	
	
	
	//Observers
	/**
	 * @return gameTypeName
	 */
	public String getGameTypeName(){
		return gameTypeName;
	}
	
	/**
	 * @return players
	 */
	public List<Player> getPlayers(){
		List<Player> tempList = new ArrayList<Player>();
		tempList.addAll(players);
		return tempList;
	}
	
	/**
	 * @return playerDescriptions
	 */
	public List<String> getPlayerDescriptions(){
		List<String> tempList = new ArrayList<String>();
		tempList.addAll(playerDescriptions);
		return tempList;
	}
	
	/**
	 * @return playerTimes
	 */
	public long[] getPlayerTimes(){
		return playerTimes;
	}
	
	/**
	 * @return allowSaveLoad
	 */
	public boolean getAllowSaveLoad(){
		return allowSaveLoad;
	}
	
}
