// -*- mode: java; indent-tabs-mode: t; c-basic-offset: 4; tab-width: 4 -*-

package antichess.test;

import antichess.*;
import junit.framework.TestCase;

/**
 * Test the MachinePlayer class.
 **/
public class MachinePlayerTests extends TestCase
{
	/**
	 * An XML &lt;piece&gt; fragment describing a test board.
	 */
	private static final String testXml = "<pieces><square id=\"a1\" piece=\"rook\" side=\"white\"/><square id=\"b1\" piece=\"knight\" side=\"white\"/><square id=\"c1\" piece=\"bishop\" side=\"white\"/><square id=\"d1\" piece=\"queen\" side=\"white\"/><square id=\"f1\" piece=\"bishop\" side=\"white\"/><square id=\"h1\" piece=\"rook\" side=\"white\"/><square id=\"a2\" piece=\"pawn\" side=\"white\"/><square id=\"b2\" piece=\"pawn\" side=\"white\"/><square id=\"c2\" piece=\"pawn\" side=\"white\"/><square id=\"d2\" piece=\"king\" side=\"white\"/><square id=\"f2\" piece=\"pawn\" side=\"white\"/><square id=\"g2\" piece=\"pawn\" side=\"white\"/><square id=\"h2\" piece=\"pawn\" side=\"white\"/><square id=\"h3\" piece=\"bishop\" side=\"black\"/><square id=\"e5\" piece=\"knight\" side=\"white\"/><square id=\"a7\" piece=\"pawn\" side=\"black\"/><square id=\"b7\" piece=\"pawn\" side=\"black\"/><square id=\"c7\" piece=\"pawn\" side=\"black\"/><square id=\"f7\" piece=\"pawn\" side=\"black\"/><square id=\"g7\" piece=\"pawn\" side=\"black\"/><square id=\"h7\" piece=\"pawn\" side=\"black\"/><square id=\"a8\" piece=\"rook\" side=\"black\"/><square id=\"b8\" piece=\"knight\" side=\"black\"/><square id=\"e8\" piece=\"king\" side=\"black\"/><square id=\"f8\" piece=\"bishop\" side=\"black\"/><square id=\"g8\" piece=\"knight\" side=\"black\"/><square id=\"h8\" piece=\"rook\" side=\"black\"/></pieces>";

	/**
	 * An XML &lt;piece&gt; fragment describing a board where white
	 * has no legal moves
	 */
	private static final String testStalemateXml = "<pieces><square id=\"f1\" side=\"white\" piece=\"king\" /> <square id=\"f2\" side=\"white\" piece=\"pawn\" /> <square id=\"f3\" side=\"black\" piece=\"king\" /> <square id=\"e5\" side=\"black\" piece=\"rook\" /> <square id=\"g5\" side=\"black\" piece=\"rook\" /></pieces>";
	
    public void testCreation() {
		new MachinePlayer(false, 0);
		new MachinePlayer(true, 0);
		new MachinePlayer(false, 0, testXml);
		new MachinePlayer(true, 0, testXml);
		new MachinePlayer(false, 0, testStalemateXml);
		new MachinePlayer(true, 0, testStalemateXml);
	}

	public void testOpeningMoves() {
		ChessMove move;
		String moveStr;
		AntichessBoard board = new AntichessBoard();
		MachinePlayer white = new MachinePlayer(true, 0);
		MachinePlayer black = new MachinePlayer(false, 0);

		board.newGame();
		moveStr = white.makeMove("", 0, 0);
		move = GameReader.parseMove(moveStr, board);
		try {
			board.doMove(move);
		} catch(IllegalMoveException e){
			fail("White made an illegal move: " + move);
		}

		moveStr = black.makeMove(moveStr, 0, 0);
		move = GameReader.parseMove(moveStr, board);
		try {
			board.doMove(move);
		} catch(IllegalMoveException e){
			fail("Black made an illegal move: " + move);
		}
	}

	public void testTestBoard() {
		ChessMove move;
		String moveStr;
		AntichessBoard board = new AntichessBoard();
		MachinePlayer player = new MachinePlayer(true, 0, testXml);

		try {
			GameReader.loadBoard(board, testXml, Player.WHITE);
		} catch(InvalidGameFileException e) {
			fail("Couldn't load sample game XML!");
		}
		moveStr = player.makeMove("", 0, 0);
		move = GameReader.parseMove(moveStr, board);
		try {
			board.doMove(move);
		} catch(IllegalMoveException e){
			fail("White made an illegal move: " + move);
		}

		player = new MachinePlayer(false, 0, testXml);
		moveStr = player.makeMove(moveStr, 0, 0);
		move = GameReader.parseMove(moveStr, board);
		try {
			board.doMove(move);
		} catch(IllegalMoveException e){
			fail("Black made an illegal move: " + move);
		}

		player = new MachinePlayer(false, 0, testXml);
		try {
			GameReader.loadBoard(board, testXml, Player.BLACK);
		} catch(InvalidGameFileException e) {
			fail("Couldn't load sample game XML!");
		}
		moveStr = player.makeMove("", 0, 0);
		move = GameReader.parseMove(moveStr, board);
		try {
			board.doMove(move);
		} catch(IllegalMoveException e){
			fail("Black made an illegal first move: " + move);
		}
	}

	/**
	 * Test that a MachinePlayer returns an empty move when it has no
	 * legal moves
	 */
	public void testStalemate() {
		String moveStr;
		MachinePlayer player = new MachinePlayer(true, 0, testStalemateXml);

		moveStr = player.makeMove("", 0, 0);
		assertEquals("Player made an empty move when he has no legal moves",
					 moveStr, "");
	}

}
