/**
 ** dl6811.c
 **
 ** Copyright 1990, 1991 by Randy Sargent.
 **
 ** The author hereby grants to MIT permission to use this software.
 ** The author also grants to MIT permission to distribute this software
 ** to schools for non-commercial educational use only.
 **
 ** The author hereby grants to other individuals or organizations
 ** permission to use this software for non-commercial
 ** educational use only.  This software may not be distributed to others
 ** except by MIT, under the conditions above.
 **
 ** Other than these cases, no part of this software may be used or
 ** distributed without written permission of the author.
 **
 ** Neither the author nor MIT make any representations about the 
 ** suitability of this software for any purpose.  It is provided 
 ** "as is" without express or implied warranty.
 **
 ** Randy Sargent
 ** Research Specialist
 ** MIT Media Lab
 ** 20 Ames St.  E15-301
 ** Cambridge, MA  02139
 ** E-mail:  rsargent@athena.mit.edu
 **
 **/


#include CONFIG

#include <confexec.h>
#include <queue.h>
#include <iob.h>
#include <board.h>

#include "dl6811.h"

Int dl_queue_size= 256;

Queue check_queue;

Int dl_debug= 0;
Int dl_ignore= 0;

Int bs_chars_received= 0;
void dl_reset_check_char(void)
{
    static initted= 0;
    if (initted) queue_term(&check_queue);
    queue_init(&check_queue, dl_queue_size, sizeof(Int));
    bs_chars_received= 0;
    initted= 1;
}

/* 0 if successful */
Int dl_check_char(IOStream *port, Int data)
{
    Int received, expected, added= 0;
    Int timeout= 500; /* milliseconds */

    if (dl_ignore) return 0;
    
    while (1) {
        if (!added) added= !queue_add_tail(&check_queue, &data);
	received= io_getchar(port, added ? 0 : 10);
	if (received == EOF && added) break;
        if (received == EOF) {
            timeout -= 10;
            if (timeout <= 0) {
                fprintf(stderr,
	       "Timeout on serial response after %ld of 256 chars received\n",
                	bs_chars_received);
		return 1;
            }
        }
	if (!queue_head(&check_queue)) {
	    fprintf(stderr,
    "Received too many characters from serial port after %ld chars received\n",
	    		bs_chars_received);
	    return 1;
	}
	expected= *(Int*)queue_remove_head(&check_queue);
        if (dl_debug) printf("GOT %lx\n", expected);
	if (expected != received) {
	    fprintf(stderr, "Expected %ld, received %ld, after %ld chars received\n", 
	    	expected, received, bs_chars_received);
	    return 1;
	}
	bs_chars_received++;
    }
    return 0;
}

Int dl_check_rest(IOStream *port)
{
    Int received, expected;

    io_flush(port);
    
    if (dl_ignore) {
	printf("Waiting for serial to finish...\n");
	msleep(5000);  /* wait for downloader to finish */
	io_discard_input(port);
	return 0;
    }
    
    while (queue_head(&check_queue)) {
	received= io_getchar(port, 500);
	if (received == EOF) {
	    fprintf(stderr, "Timeout on serial receive.  %ld chars left to receive\n",
		    queue_length(&check_queue));
	    return 1;
	}
	expected= *(Int*)queue_remove_head(&check_queue);
	if (expected != received) {
	    fprintf(stderr, "Expected %ld, received %ld\n", expected, received);
	    return 1;
	}
    }
    return 0;
}

/*****************/
/* HEX FUNCTIONS */
/*****************/

Int hex_to_int(char ch)
{
    if ('0' <= ch && ch <= '9') return ch - '0';
    if ('a' <= ch && ch <= 'f') return ch - 'a' + 10;
    if ('A' <= ch && ch <= 'F') return ch - 'A' + 10;
    printf("Illegal hex digit >%c<\n", ch);
    return 0;
}

Int hex2_to_int(char *str)
{
    return hex_to_int(str[0]) * 16 + hex_to_int(str[1]);
}

Int decode_hex(char *in, unsigned char *out)
{
    Int i;
    for (i= 0; in[0] && in[1]; i++, in+=2) {
	out[i]= (unsigned char) hex2_to_int(in);
    }
    if (in[0]) {
	printf("Uneven # of hex digits in decode_hex\n");
    }
    return i;
}

void dl_6811_help(IOStream *keyboard)
{
    printf
      ("\n");
    printf
      ("This program is attempting to download a 256 byte bootstrap into your\n"
       "6811 board.  For this to work, you must have your board hooked up to\n"
       "the correct port, and have it in download mode.\n"
       "\n");
    printf
      ("The way to get into download mode varies from board to board.  Here\n"
       "is a list of known boards and how to get them into download mode:\n"
       "\n");
    printf
      ("MIT 6.270 board, rev 2.21 (6/93)\n"
       "             Hold down the 'Choose' button while pressing RESET.  Continue\n"
       "             holding down Choose for at least 2 seconds after RESET is pressed.\n"
       "             (RESET is the large pushbutton next to the power switch)\n"
       "\n");
    printf
      ("Randy and Fred \"Mini\" (Rev 1.5) board (10/91):\n"
       "             Turn small slide switch away from processor and press RESET (XIRQ)\n"
       "             (To turn power on, turn toggle switch away from processor)\n"
       "\n");
    printf
      ("MIT 6.270 board, rev 2.0, 2.1, 2.2 (1/91-1/93), MIT Sensor Bot (9/91):\n"
       "             Power-cycle board.  Do NOT press RESET\n"
       "\n");
    printf("Press RETURN to continue...\n");
    while (1) {
    	Int c= io_getchar(keyboard, -1);
    	if (c == '\n' || c== '\r') break;
    }

    printf
    	("\n");
    printf
      ("Rug Warrior (From Mobile Robots: Inspiration to Implementation)\n");
    printf
      ("    Move large slide switch into the 'DL' position and press the\n");
    printf
      ("    reset button (the DL label is visible in small white letters above\n");
    printf
      ("    the slide switch, and the reset button is the pushbutton in the lower\n");
    printf
      ("    left-hand side of the board.\n");
    
    printf
    	("\n");
    printf
    	("Troubleshooting hints for users of the MIT 6.270 board, rev 2.21:\n"
    	 "\n");
    printf
    	("1. If the green SER RCV is not lit, you probably have a problem in the\n"
    	 "   connection between your host computer and your board.  The SER RCV\n"
    	 "   LED is powered and lit solely by the host's idle serial voltage level\n"
    	 "\n");
    printf
        ("2. Although the yellow SER XMIT light usually stays on while the power\n"
         "   switch is on, it should be OFF if you have successfully placed the board\n"
         "   in download mode.\n"
         "\n");
    
    return;
}

void dl_6811_user_friendly(IOStream *port, unsigned char *c, long len, IOStream *keyboard)
{
    Int err;
    while (1) {
        while (1) {
	    Int c;
	    printf("\n");
	    printf("Please place board in download mode and press RETURN\n");
	    printf("To quit, press Q\n");
	    printf("For help, press H\n");
	    c= io_getchar(keyboard, -1);
	    switch (c) {
	      case '\n':
	      case '\r':
	        goto out_of_loop;
	      case 'q':
	      case 'Q':
	        printf("Bye!\n");
	        exit(1);
	      case 'h':
	      case 'H':
	        dl_6811_help(keyboard);
	        break;
	      default:
	        break;
	    }
        }
      out_of_loop:;
    
	err= dl_6811(port, c, len);
	switch (err) {
	  case 0: /* no error */
	    return;
	  case 1: /* communications error */
	    printf("Perhaps the board was not connected properly or not in download mode\n");
	    break;
	  case 2:
	    exit(0);
	}
    }
}

/* 0 if successful */
/* 1 if communications error (maybe prompt user to try again) */
/* 2 if unrecoverable error */

Int dl_6811(IOStream *port, unsigned char *c, long len)
{
    long i;
    Int err;

    if (len > 256) {
	fprintf(stderr, "Attempted to bootstrap download more than 256 bytes (%ld bytes)\n", len);
	return 2;
    }
    printf("Downloading 256 byte bootstrap (%ld data)\n", len);
    io_serial_init(1200L, port);
    msleep(250);
    io_putchar(0xff, port);
    io_flush(port);
    msleep(1000);
    
    dl_reset_check_char();
    io_discard_input(port);

    for (i= 0; i< len; i++) {
    	printf("."); 
    	if (0 == ((i+1) % 64)) printf("\n"); 
    	fflush(stdout);
	io_putchar(c[i], port);
	io_flush(port);
	err= dl_check_char(port, c[i]);
	if (err) return err;
    }
    for (; i< 256; i++) {
    	printf("_");
    	if (0 == ((i+1) % 64)) printf("\n"); 
    	fflush(stdout);
	io_putchar(0, port);
	io_flush(port);
	err= dl_check_char(port, 0);
	if (err) return err;
    }
    err= dl_check_rest(port);
    if (err) return err;
    printf("Download successful\n");
    return 0;
}
    

    

