/**
 ** io_unix.c
 **
 ** Copyright 1990, 1991 by Randy Sargent.
 **
 ** The author hereby grants to MIT permission to use this software.
 ** The author also grants to MIT permission to distribute this software
 ** to schools for non-commercial educational use only.
 **
 ** The author hereby grants to other individuals or organizations
 ** permission to use this software for non-commercial
 ** educational use only.  This software may not be distributed to others
 ** except by MIT, under the conditions above.
 **
 ** Other than these cases, no part of this software may be used or
 ** distributed without written permission of the author.
 **
 ** Neither the author nor MIT make any representations about the 
 ** suitability of this software for any purpose.  It is provided 
 ** "as is" without express or implied warranty.
 **
 ** Randy Sargent
 ** Research Specialist
 ** MIT Media Lab
 ** 20 Ames St.  E15-301
 ** Cambridge, MA  02139
 ** E-mail:  rsargent@athena.mit.edu
 **
 **/


/* Serial and console library routines for UNIX */

#define IO_MODULE

#include <stdio.h>
#include <string.h>
#include <sys/types.h>
#include <sys/time.h>
#include <sys/timeb.h>
#include <fcntl.h>
#include <sys/ioctl.h>

#include "util.h"

#ifdef UNIX
#include <signal.h>
#endif

#include "die.h"

#define IOSTREAM_NAME_MAX 100

typedef struct iostream_struct {
    Int stream;
    char name[IOSTREAM_NAME_MAX];
} IOStream;

#include "io.h"

Int io_delay_factor= 1;
Int io_hex_on_serial= 0;

#ifndef DEFAULT_SERIAL_PORT
#define DEFAULT_SERIAL_PORT ""
#endif

char *io_default_serial_name = DEFAULT_SERIAL_PORT;

Int io_debug= 0;

long io_time_msec(void)
{
    struct timeb t;
    ftime(&t);
    return t.millitm + t.time * 1000L;
}

void io_set_delay_factor(Int f)
{
    if (f < 1) f= 1;
    if (f > 100) f= 100;
    io_delay_factor= f;
}

void io_set_hex_on_serial(Int hex)
{
    io_hex_on_serial= hex;
}


void io_sigint_interrupt(void)
{
    printf("Bye!\n");
#ifdef UNIX    
    system("/bin/stty cooked echo -nl -pass8");
#endif
#if defined(HAS_ATEXIT)
#elif defined(HAS_ON_EXIT)
#elif defined(UNIX)
    death_events();
    exit(0);
#endif    
}

void io_make_stdin_unbuffered(void)
{
    
#ifdef UNIX    
    system("stty cbreak -echo pass8");
#endif

#if defined(HAS_ATEXIT)
    atexit((void*) io_sigint_interrupt);
#elif defined(HAS_ON_EXIT)
    on_exit((void*) io_sigint_interrupt);
#elif defined(UNIX)
    signal(SIGINT, (void*) io_sigint_interrupt);
#endif    
    
}



/*******************/
/* SERIAL ROUTINES */
/*******************/

IOStream *io_open_serial(char *name)
{
    IOStream *ret= malloc(sizeof(IOStream));
    strncpy(ret->name, name, IOSTREAM_NAME_MAX);
    ret->name[IOSTREAM_NAME_MAX-1]= 0;
    
    ret->stream= open(ret->name, O_RDWR);

    return (ret->stream < 0 ? 0 : ret);
}

IOStream *io_open_stdin(void)
{
    IOStream *ret= malloc(sizeof(IOStream));
    ret->name[0]= 0;
    ret->stream= 0;
    return ret;
}

IOStream *io_open_fd(Int fd)
{
    IOStream *ret= malloc(sizeof(IOStream));
    ret->name[0]= 0;
    ret->stream= fd;
    return ret;
}

Int io_stream(IOStream *s)
{
    return s->stream;
}

Int io_carrier_detect(IOStream *s)
{
    Int i;
    if (ioctl(s->stream, TIOCMGET, &i) < 0) return 1;
    return i & TIOCM_CAR ? 1 : 0;
}

void io_hangup(IOStream *s)
{
    ioctl(s->stream, TIOCCDTR);
}

void io_serial_init(long baud, IOStream *s)
{
    char   execbuf[80 + IOSTREAM_NAME_MAX];
#if defined(DEC)
    sprintf(execbuf, "/bin/stty litout pass8 raw -echo even odd %ld >%s\n",
	    baud, s->name);
#elif defined(SPARC)
    sprintf(execbuf, "/bin/stty litout pass8 raw -echo -parenb %ld >%s\n",
	    baud, s->name);
#else
    sprintf(execbuf, "/bin/stty litout pass8 raw -echo %ld >%s\n",
	    baud, s->name);
#endif    
    system(execbuf);
    io_discard_input(s);
}

Int io_getchar(IOStream *s, Int timeout)
{
    IOStream *streams[2];
    Int which;
    Int ret;
    streams[0]= s;
    streams[1]= 0;
    ret= io_getchar_multiple_inputs(streams, &which, timeout);
    if (io_debug && s->stream)
      if (32 <= ret && ret <= 127)
	printf("getchar %d >%c<\n", ret, ret);
      else
	printf("getchar %d\n", ret);
    return ret;
}

void io_putstring(IOStream *s, char *foo)
{
    while (*foo) {
	io_putchar(*foo, s);
	foo++;
    }
}

void io_putchars(IOStream *s, char *foo, Int len)
{
    while (len) {
	io_putchar(*foo, s);
	foo++;
	len--;
    }
}


void io_putchar(Int c, IOStream *s)
{
    unsigned char buf= c;
    if (io_debug)
      if (32 <= c && c <= 127)
	printf("putchar %d >%c<\n", c, c);
      else
	printf("putchar %d\n", c);
    if (!io_hex_on_serial) {
	if (write(s->stream ? s->stream : 1, &buf, 1) != 1)
	  printf("ERROR ON WRITE IN IO_UNIX.C !!\n");
    } else {
	char i;
	i= (15&(c>>4))+'@';
	write(s->stream ? s->stream : 1, &i, 1);
	i= (15&(c>>0))+'@';
	write(s->stream ? s->stream : 1, &i, 1);
    }
}

Int io_getchar_multiple_inputs_n(IOStream **inputs, Int *which, Int timeout, Int n)
{
    Int nfound;
    Int i, max= 0;
    char buf;
    fd_set read_set, write_set, exceptional_set;
    struct timeval timeval_timeout;

    timeout *= io_delay_factor;

    timeval_timeout.tv_sec= (long) (timeout / 1000);
    timeval_timeout.tv_usec= 1000L * (long) (timeout % 1000);

    FD_ZERO(&read_set);
    FD_ZERO(&write_set);
    FD_ZERO(&exceptional_set);

    for (i= 0; i< n; i++) {
	FD_SET(inputs[i]->stream, &read_set);
	if (inputs[i]->stream > max) max= inputs[i]->stream;
    }

    nfound= select(max+1, &read_set, &write_set, &exceptional_set,
		   timeout < 0 ? 0 : &timeval_timeout);

    if (nfound <= 0) {
	*which= EOF;
	return EOF;
    }

    for (i= 0; i< n; i++) {
	if (FD_ISSET(inputs[i]->stream, &read_set)) {
	    *which= i;
	    if (inputs[i]->stream && io_hex_on_serial) {
		char b1;
		while (1) {
		    read(inputs[i]->stream, &b1, 1);
		    if ('@' <= b1 && b1 <= '@'+15) break;
		    printf("got %d from remote for first\n", b1);
		}
		buf= (b1-'@')<<4;
		while (1) {
		    read(inputs[i]->stream, &b1, 1);
		    if ('@' <= b1 && b1 <= '@'+15) break;
		    printf("got %d from remote for second\n", b1);
		}
		buf |= (b1-'@');
	    } else {
		if (read(inputs[i]->stream, &buf, 1) != 1) {
		    printf("Couldn't read from found filedes\n");
		}
	    }
	    return 0xff & (Int) buf;
	}
    }
    printf("Didn't find the readable file\n");
    exit(1);
    /* does not reach here */
    return 0;
}

Int io_getchar_multiple_inputs(IOStream **inputs, Int *which, Int timeout)
{
    Int n;
    for (n= 0; inputs[n]; n++);
    return io_getchar_multiple_inputs_n(inputs, which, timeout, n);
}

void io_discard_input(IOStream *s)
{
    while (io_getchar(s, 0) != EOF);
}

#ifdef DONT_HAVE_USLEEP

void io_sigint_alarm(void)
{
}

void msleep(Int mseconds)
{
    struct itimerval value;

    value.it_interval.tv_sec= 0;
    value.it_interval.tv_usec= 0;
    
    value.it_value.tv_sec= mseconds / 1000;
    value.it_value.tv_usec= (mseconds % 1000) * 1000;

    signal(SIGALRM, (void*) io_sigint_alarm);
    setitimer(ITIMER_REAL, &value, NULL);
    pause();
}

#else

void msleep(Int mseconds)
{
    usleep(io_delay_factor * 1000L * (long) mseconds);
}

#endif
    
#ifdef STAND_ALONE
NOPROTO void main(Int argc, char **argv)
{
    IOStream *kbd= io_open_stdin();
    IOStream *ser= io_open_serial(io_default_serial_name);
    IOStream *streams[3];
    Int i;

    streams[0]= kbd;
    streams[1]= ser;
    streams[2]= 0;
    
    for (i= 0; i< 3; i++) {
	Int c, which;
	
	c= io_getchar_multiple_inputs(streams, &which, 5000);
	printf("Got a char %d >%c< from %d\n", c, c, which);
    }

}
#endif

	
	
    
	
	
    
    


