/**
 ** board.c
 **
 ** Copyright 1990, 1991 by Randy Sargent.
 **
 ** The author hereby grants to MIT permission to use this software.
 ** The author also grants to MIT permission to distribute this software
 ** to schools for non-commercial educational use only.
 **
 ** The author hereby grants to other individuals or organizations
 ** permission to use this software for non-commercial
 ** educational use only.  This software may not be distributed to others
 ** except by MIT, under the conditions above.
 **
 ** Other than these cases, no part of this software may be used or
 ** distributed without written permission of the author.
 **
 ** Neither the author nor MIT make any representations about the 
 ** suitability of this software for any purpose.  It is provided 
 ** "as is" without express or implied warranty.
 **
 ** Randy Sargent
 ** Research Specialist
 ** MIT Media Lab
 ** 20 Ames St.  E15-301
 ** Cambridge, MA  02139
 ** E-mail:  rsargent@athena.mit.edu
 **
 **/


#include CONFIG

#include <util.h>
#include <util.h>
#include <boardsim.h>

#include "board.h"

#define BOARDSIM 1

#define COMPILER
#include <pcode.h>

#if BOARDSIM
#define IF_BOARDSIM(body) {if (!board_stream) {body}}
#else
#define IF_BOARDSIM(body) {}
#endif

/*-------------------------------------------------------------------------*/
/* Private Constants                                                       */

#define BOARD 0
#define KEYBOARD 1

/*-------------------------------------------------------------------------*/
/* Private Variables                                                       */

IOStream *board_stream;
IOStream *keyboard_stream;
IOStream *board_and_keyboard_streams[3];
FILE     *board_output;

/*-------------------------------------------------------------------------*/
/* Public Variables                                                        */

Int board_debug= 0;
Int board_loopback= 0;
Int board_no_return_comm= 0;

/*-------------------------------------------------------------------------*/
/* Private Functions                                                       */

void board_putchar(Int c)
{
    if (board_loopback) {
	io_getchar(board_stream, 0);
    }
    io_putchar(c, board_stream);
    if (board_loopback) {
	io_flush(board_stream);
	io_getchar(board_stream, 50);
    }
}

Int board_getnibble(Int timeout)
{
    Int c;
    io_flush(board_stream);
    while (1) {
	c= io_getchar(board_stream, timeout);
	if (c < 0) return c;
	if ('@' <= c && c <= 'O') {
	    if (board_debug) printf("got char >%c< from serial\n", (int)c);
	    return c-'@';
	}
	if (board_debug) printf("flushing char >%c< from serial\n", (int)c);
    }
}

Int board_getbyte(Int timeout)
{
#if NIBBLES
    Int datal, datah;
    datah = board_getnibble(timeout);
    if (datah < 0) return datah;
    datal = board_getnibble(timeout);
    if (datal < 0) return datal;
    return (datah << 4) + datal;
#else
    io_flush(board_stream);
    return io_getchar(board_stream, timeout);
#endif
}

long board_getword(Int timeout)
{
    Int datal, datah;
    io_flush(board_stream);
    datah= board_getbyte(timeout);
    if (datah < 0) return datah;
    datal = board_getbyte(timeout);
    if (datal < 0) return datal;
    return (datah << 8) + datal;
}

void board_putbyte(Int byte)
{
    board_putchar( byte & 255 );
}

void board_putword(Int word)
{
#if NIBBLES
    board_putchar( ((word >> 12) & 15) + '@'  );
    board_putchar( ((word >>  8) & 15) + '@'  );
    board_putchar( ((word >>  4) & 15) + '@'  );
    board_putchar( ((word >>  0) & 15) + '@'  );
#else
    board_putchar( (word >> 8) & 255 );
    board_putchar( word & 255 );
#endif    
}

Int board_special_download(unsigned char *data)
{
    Int i;
    io_serial_init(1200L, board_stream);
    board_putchar(0xff );
    io_serial_init(1200L, board_stream);
    board_putchar(data[0] );
    if ((unsigned char) io_getchar(board_stream, 100) != data[0]) {
	printf("Board not responding to special download\n");
	return 0;
    }
    for (i= 1; i< 256; i++) {
	board_putchar(data[i] );
    }
    io_flush(board_stream);
    for (i= 1; i< 256; i++) {
	if ((unsigned char) io_getchar(board_stream, 100) != data[i]) {
	    printf("Board lost data after %ld bytes in special download\n", i);
	    return 0;
	}
    }
    msleep(100);
    io_serial_init(9600L, board_stream);
    return 1;
}

void board_getprompt(void)
{
    Int c;

    if (board_no_return_comm) {
	msleep(50);
	return;
    }
				     
    io_flush(board_stream);
    c= io_getchar(board_stream, 2000);

    switch (c) {
      case EOF:
	printf("Board not responding\n");
	board_reset(0);
	break;
      case '>':
	break;
      default:
	printf("Board sync error: looking for prompt, got %ld\n", c);
	board_reset(0);
	break;
    }
}

void board_command(Int type, Int a, Int b)
{
    Int c;

    io_flush(board_stream);
    io_discard_input(board_stream); 
    while (1) {
	board_putchar(type );
	io_flush(board_stream);

	if (board_no_return_comm) {
	    msleep(50);
	    break;
	}
	else {
	    c= io_getchar(board_stream, 300);
	}
	if (c == type)
	  break;
	else if (c == EOF)
	  printf("Board not responding to %ld (%c) command\n", type, (int)type);
	else
	  printf("Board synchronization error: sent %ld, received %ld\n", type, c);
	board_reset(0);
    }
    
    board_putword(a);
    board_putword(b);
    io_flush(board_stream);
}

void board_jsr(Int pc, Int x, Int b)
{
    board_command('j', (pc << 8) + b, x);
    board_getprompt();
}
    
Int board_jsr_new(Int pc, Int d)
{
  board_command('j', d, pc);
  exit(0);
}
    
/*-------------------------------------------------------------------------*/
/* Public Functions                                                        */

Int board_read_mem(Int address)
{
    Int data= 0;
    IF_BOARDSIM(return boardsim_read_mem(address););
    board_command('r', address, address);
    data= board_getbyte(50);
    board_getprompt();
    if (board_debug) printf("read from %04lX is %02lX\n", address, data);
    return data;
}

Int board_read_mem_word(Int address)
{
    IF_BOARDSIM(return boardsim_read_mem_word(address););
    return (board_read_mem(address) << 8) + board_read_mem(address+1);
}

long board_read_mem_long(Int address)
{
    IF_BOARDSIM(return boardsim_read_mem_long(address););
    return ((long) board_read_mem_word(address+0) << 16) |
           ((unsigned long) board_read_mem_word(address+2) << 0);
}

void board_write_ram(Int address, Int data)
{
    IF_BOARDSIM(boardsim_write_ram(address, data); return;);
    if (board_debug) printf("write %02lX to %04lX\n", data, address);
    board_command('w', data, address);
    board_getprompt();
}

void board_erase_config(void)
{
    IF_BOARDSIM(return;);
    if (board_debug) printf("erase config\n");
    board_command('f', 0, 0);
    board_getprompt();
}

void board_write_eeprom(Int address, Int data)
{
    IF_BOARDSIM(return;);
    if (board_debug) printf("write %02lX to eeprom %04lX\n", data, address);
    board_command('e', data, address);
    board_getprompt();
}

void board_erase_eeprom(Int address)
{
    IF_BOARDSIM(return;);
    if (board_debug) printf("erase eeprom %04lX\n", address);
    board_command('x', 0, address);
    board_getprompt();
}

void board_clear_and_set_ram(Int address, Int clear, Int set)
{
    IF_BOARDSIM(boardsim_clear_and_set_ram(address, clear, set); return;);
    board_command('c', ((0xff & ~clear) << 8) | (0xff & set), address);
    board_getprompt();
}

Int board_process_wait(void)
{
    Int which, c;
    IF_BOARDSIM(return boardsim_process_wait(););
    io_flush(board_stream);
    while (1) {
	c= io_getchar_multiple_inputs(board_and_keyboard_streams, &which, -1);
	if (which == 0) {
	    if (c == '\n' || c == '\r') {
		return 0;
	    } else {
		printf("Press <return> to stop\n");
	    }
	} else {
	    return 1;
	}
    }
}

void board_write_ram_word(Int address, Int data)
{
    IF_BOARDSIM(boardsim_write_ram_word(address, data); return;);
    board_write_ram(address, (data >> 8) & 0xff);
    board_write_ram(address+1, data & 0xff);
}
    
Int board_set_streams(IOStream *board, IOStream *keyboard)
{
    if (board) io_serial_init(9600L, board);
#if !BOARDSIM
    else return 0;
#endif    

    board_and_keyboard_streams[0]= keyboard_stream= keyboard;
    board_and_keyboard_streams[1]= board_stream= board;
    board_and_keyboard_streams[2]= 0;

    return 1;
}

void board_set_output(FILE *output)
{
    board_output= output;
}

void board_download_block(Int start, Int len, unsigned char *data)
{
    Int checksum, i;
    IF_BOARDSIM(boardsim_download_block(start, len, data); return;);
    board_command('b', len, start);
    if (board_debug) printf("[");
    for (i= 0; i< len; i++) {
	if (board_debug) printf("%02X ", data[i]);
	board_putbyte(data[i]);
    }
    if (board_debug) printf("]\n");
    io_flush(board_stream);
    checksum= board_getbyte(1000);
    UNUSED(checksum);
    board_getprompt();
}
    
void board_download(Int start, Int len, unsigned char *data, Int display)
{
    char buf[10];
    Int block, j, nblocks, offset;
    
    IF_BOARDSIM(boardsim_download(start, len, data, display); return;);
      
    if (display) printf("Downloading %ld bytes (addresses %04lX-%04lX):  ",
			len, start, start + len - 1);

    nblocks = (len + 99) / 100;

    for (block= 0, offset= 0; block < nblocks; block++, offset += 100) {
	Int blocklen= 100;
	if (blocklen + offset > len) blocklen= len - offset;

	sprintf(buf, "%ld", offset);

	board_download_block(start+offset, blocklen, data + offset);

	if (display) {
	    printf("%s", buf);
	    for (j= strlen(buf); j; j--) putchar(8);
	}
	fflush(stdout);
    }
    
    if (display) printf("%ld loaded\n", len);
}

void board_reset(Int first_time)
{
    Int timeout= 0, c, annoyed= 0;

    IF_BOARDSIM(boardsim_reset(first_time); return;);
    
    printf("Synchronizing with board\n");
    io_flush(board_stream);
    msleep(10);
    io_discard_input(board_stream);
    while(1) {
	if (!(++timeout % 10)) {
	    printf("Board disconnected or not responding, retrying.");
	    if (first_time && !annoyed) {
		/*printf("  Did you remember to press the reset button?");*/
	    }
	    annoyed++;
		/* if (annoyed > 3) printf("  ()."); */
	    printf("\n");
	}
	board_putchar('s' );
	io_flush(board_stream);
	while ((c= io_getchar(board_stream, 200)) != EOF) {
	    if (c == 's') goto got_it;
	}
        c= io_getchar(keyboard_stream, 0);
        if (c == 3) exit(0);
    }
  got_it:
    msleep(50);
    board_getprompt();
}

void board_kill_all_processes(void)
{
    Int i;

    board_clear_and_set_ram(0x0b, 0, SYSSTAT_ERRORHALT);

    for (i= PROCESS_TABLE + PROCESS_SLOT_LENGTH;
	 i< PROCESS_TABLE + MAX_PROCESSES * PROCESS_SLOT_LENGTH;
	 i += PROCESS_SLOT_LENGTH) {
	board_write_ram(i+P_STATUS, PSTAT_DEAD);
    }
    
    board_write_ram_word(PROCESS_TABLE + P_PREV, PROCESS_TABLE);
    board_write_ram_word(PROCESS_TABLE + P_NEXT, PROCESS_TABLE);
    board_clear_and_set_ram(0x0b, SYSSTAT_ERRORHALT, 0);
}

void board_get_version(Int *major, Int *minor)
{
    IF_BOARDSIM(boardsim_get_version(major, minor); return;);
    *major= board_read_mem(VERSION_NUMBER);
    *minor= board_read_mem(VERSION_NUMBER+1);
}

Int board_get_msb_first(void)
{
    IF_BOARDSIM(return boardsim_get_msb_first(););
    return 1;
}

Int board_get_float_type(void)
{
    IF_BOARDSIM(return boardsim_get_float_type(););
    return float_6811;
}

