/**
 ** iob_unix.c
 **
 ** Copyright 1990, 1991 by Randy Sargent.
 **
 ** The author hereby grants to MIT permission to use this software.
 ** The author also grants to MIT permission to distribute this software
 ** to schools for non-commercial educational use only.
 **
 ** The author hereby grants to other individuals or organizations
 ** permission to use this software for non-commercial
 ** educational use only.  This software may not be distributed to others
 ** except by MIT, under the conditions above.
 **
 ** Other than these cases, no part of this software may be used or
 ** distributed without written permission of the author.
 **
 ** Neither the author nor MIT make any representations about the 
 ** suitability of this software for any purpose.  It is provided 
 ** "as is" without express or implied warranty.
 **
 ** Randy Sargent
 ** Research Specialist
 ** MIT Media Lab
 ** 20 Ames St.  E15-301
 ** Cambridge, MA  02139
 ** E-mail:  rsargent@athena.mit.edu
 **
 **/


/* Serial and console library routines for UNIX */

#include <stdio.h>
#include <string.h>
#include <sys/types.h>
#include <sys/time.h>
#ifdef HAS_SYS_TIMEB_H
#include <sys/timeb.h>
#endif
#include <fcntl.h>
#include <sys/ioctl.h>

#ifdef USE_TERMIOS
#ifdef HAS_TERMIOS_H
#include <termios.h>
#endif
#ifdef HAS_SYS_TERMIOS_H
#include <sys/termios.h>
#endif
#else

#ifdef HAS_SGTTY_H
#include <sgtty.h>
#endif
#endif

#include <signal.h>

#include "util.h"
#include "byteq.h"
#include "stringlb.h"
#include "filelib.h"
#include "growbuf.h"
#include "iob.h"


void io_sigint_interrupt(void)
{
    printf("Bye!\n");
#ifdef USE_TERMIOS
    restore_settings();
#else    
#if defined(UNIX) && !defined(SOLARIS)
    system("/bin/stty cooked echo -nl -pass8");
#elif defined(SOLARIS)
    system("/bin/stty cooked echo -nl");
#endif
#endif    
    exit(0);
}

#ifdef TIOCMGET
#define GET_MODEM_STATE TIOCMGET
#define SET_MODEM_STATE TIOCMSET
#else
#ifdef TIOCMODG
#define GET_MODEM_STATE TIOCMODG
#define SET_MODEM_STATE TIOCMODS
#endif
#endif

#ifdef GET_MODEM_STATE

void io_set_rts(IOStream *s, int val)
{
   /*int state;*/
   /*ioctl(s->stream, GET_MODEM_STATE, &state);*/
   if (val) ioctl(s->stream, TIOCMBIS, TIOCM_RTS);
   else ioctl(s->stream, TIOCMBIC, TIOCM_RTS);
#if 0   
   if (val) state |= TIOCM_RTS; else state &= ~TIOCM_RTS;
   ioctl(s->stream, SET_MODEM_STATE, &state);
#endif   
}

int io_get_cts(IOStream *s)
{
   int state;
   ioctl(s->stream, GET_MODEM_STATE, &state);
   return !!(state & TIOCM_CTS);
}

void io_set_dtr(IOStream *s, int val)
{
   int state;
   ioctl(s->stream, GET_MODEM_STATE, &state);
   if (val) state |= TIOCM_DTR; else state &= ~TIOCM_DTR;
   ioctl(s->stream, SET_MODEM_STATE, &state);
}

#endif



#ifdef USE_TERMIOS
int     old_settings_initted;
Growbuf old_settings;

void record_settings(int fd)
{
   struct termios tio;

   tcgetattr(fd, &tio);
   if (!old_settings_initted) {
      growbuf_init(&old_settings);
      old_settings_initted= 1;
   }
   growbuf_add_int(&old_settings, fd);
   growbuf_add(&old_settings, &tio, sizeof(tio));
   /*printf("Recorded %d\n", fd);*/
}

void restore_settings(void)
{
   struct termios tio;
   int fd;

   if (old_settings_initted) {
      while (GROWBUF_LENGTH(&old_settings) > 0) {
	 growbuf_remove(&old_settings, &tio, sizeof(tio));
	 fd= growbuf_remove_int(&old_settings);
	 
	 tcsetattr(fd, TCSANOW, &tio);
	 /*printf("Restored %d\n", fd);*/
      }
   }
}
   
void io_make_raw(IOStream *s)
{
   
   make_fd_raw(s->stream, 0);
}

void io_make_stdin_unbuffered(void)
{
   make_fd_raw(0, 0);
}

void make_fd_raw(int fd, int pass_all)
{
   struct termios tio;

   record_settings(fd);
   
   tcgetattr(fd, &tio);

   tio.c_iflag= ICRNL | IXON | BRKINT ;
   tio.c_lflag = ISIG | IEXTEN ;
   tio.c_oflag = OPOST | ONLCR  ;

   tio.c_cc[VTIME]= 0;
   tio.c_cc[VMIN]= 0;

   if (pass_all)
     bzero(&tio.c_cc, sizeof(tio.c_cc)); /* clear control characters */

   tcsetattr(fd, TCSANOW, &tio);

   signal(SIGINT, (void*) io_sigint_interrupt);
}

void io_make_raw_pass_all(IOStream *s)
{
   make_fd_raw(s->stream, 1);
}

#else

void io_make_stdin_unbuffered(void)
{
   /*record_settings(stdin);*/
   
   /*my_on_exit(io_restore_stdin);*/
   
#if defined(SOLARIS)    
    system("stty raw -echo isig -echoe opost");
#else
    system("stty cbreak -echo pass8");
#endif

#if 0
#if defined(HAS_ATEXIT)
    atexit((void*) io_sigint_interrupt);
#elif defined(HAS_ON_EXIT)
    on_exit((void*) io_sigint_interrupt);
#endif
#endif
   
#if defined(UNIX)
    signal(SIGINT, (void*) io_sigint_interrupt);
#endif
    
}

void io_make_raw_pass_all(IOStream *s)
{
    if (s->stream == 0) {
	system("stty cbreak -echo pass8 intr undef susp undef eof undef");
    }
    else {
	system(free_soon(string_printf("stty raw -echo pass8 intr undef susp undef eof undef > %s", s->name)));
    }
}

void io_make_raw(IOStream *s)
{
    if (s->stream == 0) {
	system("stty cbreak -echo pass8");
    }
    else {
	system(free_soon(string_printf("stty cbreak -echo pass8 > %s", s->name)));
    }
#if 0
#if defined(HAS_ATEXIT)
    atexit((void*) io_sigint_interrupt);
#elif defined(HAS_ON_EXIT)
    on_exit((void*) io_sigint_interrupt);
#endif
#endif
#ifdef UNIX
    signal(SIGINT, (void*) io_sigint_interrupt);
#endif    
}

#endif

/* Returns 0 if drained.
   -1 if OS is incapable of draining */

int io_drain(IOStream *ios)
{
#ifdef HAS_TCDRAIN
   int time1= mtime();
   tcdrain(ios->stream);
   printf("tcdrain took %d msec\n", mtime() - time1);
   return 0;
#else
   return -1;
#endif
}

/*******************/
/* SERIAL ROUTINES */
/*******************/

IOStream *io_open_serial(char *name)
{
    IOStream *ret= malloc(sizeof(IOStream));
    strncpy(ret->name, name, IOSTREAM_NAME_MAX);
    ret->name[IOSTREAM_NAME_MAX-1]= 0;
#ifdef O_NDELAY
    /* Open with NDELAY to make sure we don't hang
       because of handshaking lines! */
    ret->stream= open(ret->name, O_RDWR | O_NDELAY);
    if (ret->stream >= 0) {
       /* Now take the O_NDELAY back off so we actually wait
	  for buffers to flush when we do writes */
       fcntl(ret->stream, F_SETFL, ~O_NDELAY & fcntl(ret->stream, F_GETFL));
    }
#else
    ret->stream= open(ret->name, O_RDWR);
#endif
    bytequeue_init(&ret->inbuf, 1020);
    bytequeue_init(&ret->outbuf, 1021);
    ret->extended_charset= 0;
    ret->prefix_detected= 0;
    return (ret->stream < 0 ? 0 : ret);
}

IOStream *io_open_filename(char *name)
{
    IOStream *ret= malloc(sizeof(IOStream));
    strncpy(ret->name, name, IOSTREAM_NAME_MAX);
    ret->name[IOSTREAM_NAME_MAX-1]= 0;
    
    ret->filestream= fopen(ret->name, "r+");
    if (!ret->filestream) return 0;
    ret->stream= fileno(ret->filestream);
    bytequeue_init(&ret->inbuf,1024);
    bytequeue_init(&ret->outbuf, 1024);
    ret->extended_charset= 0;
    ret->prefix_detected= 0;
    return (ret->stream < 0 ? 0 : ret);
}

IOStream *io_open_fd(Int fd)
{
    IOStream *ret= malloc(sizeof(IOStream));

    sprintf(ret->name, "<fd %ld>", fd);
    ret->name[IOSTREAM_NAME_MAX-1]= 0;
    
    ret->filestream= fdopen(fd, "r+");
    ret->stream= fd;
    bytequeue_init(&ret->inbuf, 1024);
    bytequeue_init(&ret->outbuf, 1024);
    ret->extended_charset= 0;
    ret->prefix_detected= 0;
    return ret;
}

IOStream *io_open_stdin(void)
{
    IOStream *ret= malloc(sizeof(IOStream));
    ret->name[0]= 0;
    ret->stream= 0;
    bytequeue_init(&ret->inbuf, 1022);
    bytequeue_init(&ret->outbuf, 1023);
    ret->extended_charset= 0;
    ret->prefix_detected= 0;
    return ret;
}

IOStream *io_open_window(void)
{
    char buf[300];
    char filname[200];
    FILE *f;
    tmpnam(filname);
    sprintf(buf, "xterm -e csh -f -c 'tty > %s; sleep 1000000' &", filname);
    /*debugf("doing sys >%s<\n", buf);*/
    system(buf);
    /*debugf("after sys, looking for file %s\n", filname);*/
    
    while (1) {
	msleep(500);
	f= fopen(filname, "r");
	if (!f) {
	    /*debugf("couldn't open");*/
	    continue;
	}
	buf[0]= 0;
	fgets(buf, 1000, f);
	fclose(f);
	if (!buf[0]) {
	    /*debugf("could open, no data");*/
	    continue;
	}
	break;
    }
    if (strchr(buf, '\n')) *strchr(buf, '\n')= 0;
    if (strchr(buf, '\r')) *strchr(buf, '\r')= 0;
    filenm_delete(filname);
    return io_open_filename(buf);
}

Int io_setxy(IOStream *s, Int x, Int y)
{
    if (io_disable_xy) return 0;
    io_printf(s, "\033[%d;%dH", y+1, x+1);
    return 0;
}

void io_highlight_video(IOStream *s, Int on)
{
    io_putstring(s, on ? "\033[4m" : "\033[m");
}

void io_inverse_video(IOStream *s, Int on)
{
    io_putstring(s, on ? "\033[7m" : "\033[m");
}

Int io_clearscreen(IOStream *s)
{
    if (io_disable_xy) return 0;
    io_putstring(s, "\033[H\033[2J");
    io_flush(s);
    return 0;
}

Int io_stream(IOStream *s)
{
    return s->stream;
}

#if 0
Int io_carrier_detect(IOStream *s)
{
    Int i;
    if (ioctl(s->stream, TIOCMGET, &i) < 0) return 1;
    return i & TIOCM_CAR ? 1 : 0;
}

void io_hangup(IOStream *s)
{
    ioctl(s->stream, TIOCCDTR);
}
#else
Int io_carrier_detect(IOStream *s)
{
    return 1;
}

void io_hangup(IOStream *s)
{
}
#endif

#ifdef USE_TERMIOS

int baud2B(long baud)
{
   switch (baud) {
    case 300: return B300;
    case 1200: return B1200;
    case 2400: return B2400;
    case 4800: return B9600;
    case 9600: return B9600;
#ifdef B19200
    case 19200: return B19200;
#else
    case 19200: return EXTA;
#endif
#ifdef B38400
    case 38400: return B38400;
#else
    case 38400: return EXTB;
#endif
    default: fprintf(stderr, "don't know baud %ld\n", baud);
      return B9600;
   }
}
      
void io_serial_init(long baud, IOStream *s)
{
   struct termios tio;
   int fd= s->stream;
   
   record_settings(fd);

   tcgetattr(fd, &tio);

   tio.c_iflag= IGNBRK;
   tio.c_oflag= 0;
   tio.c_cflag= baud2B(baud) | CS8 | CREAD | CLOCAL;
   tio.c_lflag= 0;
   
   bzero(&tio.c_cc, sizeof(tio.c_cc)); /* clear control characters */

   tcsetattr(fd, TCSADRAIN, &tio);
   
   io_discard_input(s);
}

#else
void io_serial_init(long baud, IOStream *s)
{
    char   execbuf[80 + IOSTREAM_NAME_MAX];
#if defined(DEC)
    sprintf(execbuf, "/bin/stty litout pass8 raw -echo even odd %ld >%s\n",
	    baud, s->name);
#elif defined(SOLARIS)
    sprintf(execbuf, "/bin/stty raw cs8 -ignbrk -inpck -loblk clocal -cstopb -hup cread -echo -opost -echoe -echok  -ixon -ixoff -iexten %ld < %s",
	    baud, s->name);
#elif defined(SPARC)
    sprintf(execbuf, "/bin/stty litout pass8 raw -echo -parenb %ld >%s\n",
	    baud, s->name);
#else
    sprintf(execbuf, "/bin/stty litout pass8 raw -echo %ld >%s\n",
	    baud, s->name);
#endif    
    system(execbuf);
    io_discard_input(s);
}
#endif

char *io_fn_prefix="\033[";

Int io_getchar_multiple_inputs_n(IOStream **inputs, Int *which, Int timeout, Int n)
{
    IOStream *input;
    Int c;
    while (1) {
	c= io_getchar_multiple_inputs_n_internal(inputs, which, timeout, n);
	/*printf("[%d]", c);*/
	if (c == EOF) return c;
	input= inputs[*which];
	if (!input->extended_charset) {
	    return c;
	}
	/*printf("<%d, prefix=%d>", c, input->prefix_detected);*/
	if (input->prefix_detected == 2) {
	    input->prefix_detected= 0;
	    switch (c) {
	      case 'A': return IO_UPARROW;
	      case 'B': return IO_DOWNARROW;
	      case 'C': return IO_RIGHTARROW;
	      case 'D': return IO_LEFTARROW;
	      default:	return 27; /* not quite the right thing to do */
	    }
	}
	if (io_fn_prefix[input->prefix_detected] != c) {
	    input->prefix_detected= 0;
	    return c;
	}
	input->prefix_detected++;
    }
}

Int io_write(IOStream *s, char *buf, long len)
{
    long wrote;
    long pos= 0;
    /* Loop until all written out */
    while (pos < len) {
	wrote= write (s->stream ? s->stream : 1, buf+pos, len-pos);
	if (wrote < 0) {
	    fprintf(stderr, "Error in io_write\n");
	    return -1;
	}
	pos += wrote;
    }
    return len;
}

Int io_getchar_multiple_inputs_n_internal(IOStream **inputs, Int *which, Int timeout, Int n)
{
    Int nfound;
    Int i, max= 0;
    char buffer[1024];
    fd_set read_set, write_set, exceptional_set;
    struct timeval timeval_timeout;

    for (i= 0; i< n; i++) {
	if (!BYTEQUEUE_EMPTY(&inputs[i]->inbuf)) {
	    *which= i;
	    return bytequeue_remove_char(&inputs[i]->inbuf);
	}
    }
    
    timeout *= io_delay_factor;

    timeval_timeout.tv_sec= (long) (timeout / 1000);
    timeval_timeout.tv_usec= 1000L * (long) (timeout % 1000);

    FD_ZERO(&read_set);
    FD_ZERO(&write_set);
    FD_ZERO(&exceptional_set);

    for (i= 0; i< n; i++) {
	FD_SET(inputs[i]->stream, &read_set);
	if (inputs[i]->stream > max) max= inputs[i]->stream;
    }

    nfound= select(max+1, &read_set, &write_set, &exceptional_set,
		   timeout < 0 ? 0 : &timeval_timeout);

    if (nfound <= 0) {
	*which= EOF;
	return EOF;
    }

    for (i= 0; i< n; i++) {
	if (FD_ISSET(inputs[i]->stream, &read_set)) {
	    Int nread;
	    *which= i;
	    nread= read(inputs[i]->stream, buffer, BYTEQUEUE_SPACE(&inputs[i]->inbuf));
	    if (nread <= 0) {
		/* *which= EOF; */
		return EOF;
	    }
	    else {
		/*if (inputs[i]->stream != 0) {
		    write(1, buffer, nread);
		}*/

		bytequeue_add(&inputs[i]->inbuf, buffer, nread);
	    }
	    return bytequeue_remove_char(&inputs[i]->inbuf);
	}
    }
    printf("Didn't find the readable file\n");
    exit(1);
    /* does not reach here */
    return 0;
}

void msleep(Int mseconds)
{
    struct timeval timeval_timeout;
    fd_set fd1, fd2, fd3;

    if (mseconds < 1) return;

    FD_ZERO(&fd1);
    FD_ZERO(&fd2);
    FD_ZERO(&fd3);
    
    timeval_timeout.tv_sec= (long) (mseconds / 1000);
    timeval_timeout.tv_usec= 1000L * (long) (mseconds % 1000);

    select(1, &fd1, &fd2, &fd3, &timeval_timeout);
}

#if 0

void msleep(Int mseconds)
{
    usleep(io_delay_factor * 1000L * (long) mseconds);
}

#endif


#ifdef HAS_GETTIMEOFDAY
long mtime(void)
{
    struct timeval tv;
    struct timezone tz;
    
    gettimeofday(&tv, &tz);

    return (tv.tv_sec * 1000L) + (tv.tv_usec / 1000L);
}
#else
long mtime(void)
{
    struct timeb t;
    ftime(&t);
    return t.millitm + t.time * 1000L;
}
#endif

long io_time_msec(void)
{
   return mtime();
}


	
	
    
	
	
    
    

