#!/usr/local/bin/perl
#  engine.pl - the CBB 'engine'.
#              This script implements a transaction abstract data type
#              It encapsulates a list a transactions and the functions
#              required to manipulate the transactions.
#
#  Written by Curtis Olson.  Started August 22, 1994.
#
#  Copyright (C) 1994, 1995, 1996  Curtis L. Olson  - curt@sledge.mn.org
#
#  This program is free software; you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation; either version 2 of the License, or
#  (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program; if not, write to the Free Software
#  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

# $Id: engine.pl,v 1.32 1996/02/11 20:23:58 curt Exp $
# (Log is kept at end of this file)


# specify the installed location of the necessary pieces.
$cbb_incl_dir = ".";
unshift(@INC, $cbb_incl_dir);


require "common.pl";
require "log.pl";


$| = 1;				# flush buffers after every write

if ( $logging != 0 && $logging != 1) {
    # if not specified elsewhere, turn on logging
    $logging = 1;			# 0 = off,  1 = on
}

if ( $debug != 0 && $debug != 1) {
    # if not specified elsewhere, turn off debugging.
    $debug = 0;			# 0 = off,  1 = on
}

# Global variables

# %TRANS - an associative array of transactions and transaction keys
# @KEYS - a sorted list of transaction keys (for traversing the trans list)
# $sorted_keys - specifies whether the list in @KEYS is valid
# $calced - specified whether the transactions have been properly calculated
# $current - specifies the "current" position in the @KEYS array
# $current_file - full name of currently opened transaction file
# %BALS - an associative array used to store account information
# $version - version number (set in common.pl)

open(DEBUG, ">debug") if $debug;

&init_trans();			# initialize %TRANS, @KEYS, and $sorted_keys


# get next available key for a specified date
sub get_next_key {
    # in: date
    # out: key

    local($date) = @_;
    $count = 0;

    while ( $TRANS{"$date-".&pad($count)} ) {
	$count++;
    }

    return "$date-".&pad($count);
}


# set @KEYS = sorted list of transaction keys
sub sort_keys {
    $sorted_keys = 1;
    $current = 0;

    print DEBUG "sort_keys()\n" if $debug;
    @KEYS = sort(keys %TRANS);
}


# recalculate the transactions
sub calc_trans {
    $calced = 1;
    local($total, $ntotal, $stotal, $ctotal) = (0.00, 0.00, 0.00, 0.00);
    local($count, $ncount, $scount, $ccount) = (0, 0, 0, 0);

    print DEBUG "calc_trans()\n" if $debug;

    if ($sorted_keys == 0) {
	&sort_keys();
    }

    foreach $key (@KEYS) {
        ($date, $check, $desc, $debit, $credit, $cat, $com, $cleared, 
        	$junk) = split(/\t/, $TRANS{$key});

	$total = $total + $credit - $debit;
	$count++;

	if ( ($cleared eq "x") || ($cleared eq "X") ) {
	    $ctotal = $ctotal + $credit - $debit;
	    $ccount++;
	} elsif ( $cleared eq "*" ) {
	    $stotal = $stotal + $credit - $debit;
	    $scount++;
	} else {
	    $ntotal = $ntotal + $credit - $debit;
	    $ncount++;
	}

	$TRANS{$key} = 
	  "$date\t$check\t$desc\t$debit\t$credit\t$cat\t$com\t$cleared\t".
	  	sprintf("%.2f", $total);
    }

    $BALS{"Amount"} =  $total;
    $BALS{"Count"} =   $count;

    $BALS{"Xamount"} = $ctotal;
    $BALS{"Xcount"} =  $ccount;

    $BALS{"*amount"} = $stotal;
    $BALS{"*count"} =  $scount;

    $BALS{"Namount"} = $ntotal;
    $BALS{"Ncount"} =  $ncount;
}


# create a transaction (and add to the transaction list)
sub create_trans {
    # in: transaction
    # out: keyed_transaction

    local($trans) = @_;
    $sorted_keys = 0;
    $calced = 0;

    ($date, $check, $desc, $debit, $credit, $cat, $com, $cleared, $total) =
	split(/\t/, $trans);

    if ( length($date) == 6 ) {
	$date = "19$date";
    }

    $key = &get_next_key($date);

    $trans = "$date\t$check\t$desc\t$debit\t$credit\t$cat\t$com\t$cleared\t$total";

    $TRANS{$key} = "$trans";

    print DEBUG "created:  $key\t$trans\n" if $debug;

    return "$key\t$trans";
}


# create a transfer transaction in the current file and the transfer to file
sub create_xfer {
    # in: transaction
    # out: keyed_transaction

    local($trans) = @_;
    local($orig_file) = $current_file;
    local($to_file, $from_cat);

    $sorted_keys = 0;
    $calced = 0;

    ($date, $check, $desc, $debit, $credit, $cat, $com, $cleared, $total) =
	split(/\t/, $trans);

    print DEBUG "(xfer) current_file = $current_file\n" if $debug;
    # determine the "from" category
    $from_cat = "[".&file_basename(&file_root($current_file))."]";

    # determine the "to" file name
    $to_file = $cat;
    chop($to_file);
    $to_file = substr($to_file, 1);
    $to_file = &file_dirname($current_file)."/$to_file";
    print DEBUG "to file = '$to_file' ($to_file.cbb)\n" if $debug;
    if ( -e "$to_file.cbb" ) {
	$to_file .= ".cbb";
    } elsif ( -e "$to_file.dir" ) {
	$to_file .= ".dir";
    } else {
	return "error";
    }

    print DEBUG "Transfer to $to_file\n" if $debug;

    # create the "to" transaction
    $to_trans = "$date\t$check\t$desc\t".$credit."\t".$debit."\t".
		$from_cat."\t$com\t$cleared\t$total";

    if ( &file_extension($to_file) eq "dir" ) {
	if ( &file_extension($orig_file) eq "cbb" ) {
	    # we need special handling here to preserve the .cbb file
	    # save the current transactions to a temporary file
	    $result = &save_trans("$orig_file.$$.tmp");
	    return "error" if ( $result eq "error" );
            %TRANS = ();  # clear out any transactions from the current file
	}

        # open the "to" account
        $result = &load_trans($to_file);
	return "error" if ( $result eq "error" );

	$result = &create_trans($to_trans);

	if ( &file_extension($orig_file) eq "cbb" ) {
	    $result = &load_cbb_trans("$orig_file.$$.tmp");
	    return "error" if ( $result eq "error" );
	    unlink("$orig_file.$$.tmp");
	    $current_file = $orig_file;
	} else {
            # re-open orig account
            $result = &load_trans($orig_file);
	    return "error" if ( $result eq "error" );
        }
    } else {
	$result = &append_trans($to_file, $to_trans);
	return "error" if ( $result eq "error" );
    }

    # create the "from" transaction
    $result = &create_trans($trans);

    return "$key\t$trans";
}


# append a transaction to the specified file
sub append_trans {
    local($to_file, $to_trans) = @_;

    print DEBUG "xfer to $to_file: $to_trans\n" if $debug;

    open(TOFILE, ">>$to_file") || return "error";
    print TOFILE "$to_trans\n";
    close(TOFILE);

    return "$to_trans";
}


# update a transaction (replace in the transaction list)
sub update_trans {
    # in: keyed_transaction
    # out: keyed_transaction

    local($keyed_trans) = @_;
    $sorted_keys = 0;
    $calced = 0;

    ($key, $trans) = split(/\t/, $keyed_trans, 2);

    &delete_trans($key);
    $result = &create_trans($trans);

    print DEBUG "updated:  $key\n" if $debug;
    print DEBUG "     to:  $result\n" if $debug;

    return "$result";
}


# delete a transaction given the key
sub delete_trans {
    # in: key

    local($key) = @_;
    $sorted_keys = 0;
    $calced = 0;

    delete $TRANS{$key};

    if ($current > 0) {
	--$current;
    }

    print DEBUG "deleted:  $key\n" if $debug;

    return "ok";
}


# return the next transaction
sub next_trans {
    if ($sorted_keys == 0) {
	&sort_keys();
    }

    if ($calced == 0) {
	&calc_trans();
    }

    ++$current;
    $trans = $TRANS{$KEYS[$current]};
    if ( $trans ) {
        return "$KEYS[$current]\t$trans";
    } else {
        return "none";
    }
}


# return the transaction specified by a key
sub find_trans {
    # uses a binary search so that we can keep $current current.   
    # Yeeeks! I have to think for a change.
    # Hmmm, maybe I should rethink my data structures ... nah. :)

    local($key) = @_;
    local($left, $middle, $right) = (0, 0, $#KEYS);

    if ($sorted_keys == 0) {
	&sort_keys();
    }

    if ($calced == 0) {
	&calc_trans();
    }

    $trans = "";

    while ( $left <= $right ) {
	$middle = int( ($left + $right) / 2 );
        print DEBUG "$left < $middle < $right\n" if $debug;
	if ( $KEYS[$middle] lt $key ) {
	    $left = $middle + 1;
	    print DEBUG "  left = middle + 1\n" if $debug;
        } elsif ( $KEYS[$middle] gt $key ) {
	    $right = $middle - 1;
	    print DEBUG "  right = middle - 1\n" if $debug;
        } else {
	    # we found it, set $trans to what we want and force an exit of
	    # the while loop
	    $trans = $TRANS{$KEYS[$middle]};
	    print DEBUG "  found it: $trans\n" if $debug;
	    $current = $middle;
	    $left = $right + 1;
        }
    }

    print DEBUG "found:  $key\t$trans\n" if $debug;

    if ( $trans ) {
        return "$key\t$trans";
    } else {
        return "none";
    }
}


# returns the current index
sub get_current_index {
    return $current;
}


# return the first transaction
sub first_trans {
    if ($sorted_keys == 0) {
	&sort_keys();
    }

    if ($calced == 0) {
	&calc_trans();
    }

    $current = 0;
    $trans = $TRANS{$KEYS[$current]};
    if ( $trans ) {
        return "$KEYS[$current]\t$trans";
    } else {
        return "none";
    }
}


# returns the entire transaction list in one big chunk.
sub all_trans {
    # in: date
    # out: result

    local($date_fmt) = @_;

    $| = 0;				# turn off buffer flushing

    if ($calced == 0) {
	&calc_trans();
    }

    if ($sorted_keys == 0) {
	&sort_keys();
    }

    foreach $key (@KEYS) {
	# print ("$key\t$TRANS{$key}\n");
        ($date, $check, $desc, $debit, $credit, $cat, $com, $cleared, $total) =
	    split(/\t/, $TRANS{$key});

        if ( length($date) == 6 ) {
            ($year, $month, $day) = $date =~ /(\d\d)(\d\d)(\d\d)/;
	    $year = "19" . $year;
        } else {
            ($year, $month, $day) = $date =~ /(\d\d\d\d)(\d\d)(\d\d)/
        }

        $checklen = length($check);
        if ( $checklen > 5 ) {
            $cutcheck = substr($check, $checklen - 5, 5);
        } else {
            $cutcheck = $check;
        }

        if ( $date_fmt == 1 ) {
            $nicedate = "$month/$day/" . substr($year, 2, 2);
        } else {
            $nicedate = "$day/$month/" . substr($year, 2, 2);
        }

        $cutdesc = substr($desc, 0, 15);
        $cutcom = substr($com, 0, 15);
        if ( $cat =~ m/\|/ ) {
            $nicecat = "-Splits-";
        } else {
            $nicecat = $cat;
        }
	$nicecat = substr($nicecat, 0, 9);

        printf("%5s  %-8s  %-15s  %9.2f  %9.2f  %-1s  %9.2f %14s\n",
                $cutcheck, $nicedate, $cutdesc, $debit, $credit, $cleared, 
		$total, $key);
        printf("%5s  %-8s  %-15s  %-9s %39s\n", "", "", $cutcom, $nicecat, 
		$key);
    }

    $| = 1;				# turn buffer flushing back on

    return "none";
}

# return the first uncleared transaction
sub first_uncleared_trans {
    if ($sorted_keys == 0) {
	&sort_keys();
    }

    if ($calced == 0) {
	&calc_trans();
    }

    $current = 0;
    $trans = $TRANS{$KEYS[$current]};
    ($date, $check, $desc, $debit, $credit, $cat, $com, $cleared, $junk) = 
    	    split(/\t/, $trans);
    while ( $cleared eq "x" ) {
        ++$current;
        $trans = $TRANS{$KEYS[$current]};
        ($date, $check, $desc, $debit, $credit, $cat, $com, $cleared, $junk) = 
    	        split(/\t/, $trans);
    }

    if ( $trans ) {
        return "$KEYS[$current]\t$trans";
    } else {
        return "none";
    }
}


# return the next uncleared transaction
sub next_uncleared_trans {
    if ($sorted_keys == 0) {
	&sort_keys();
    }

    if ($calced == 0) {
	&calc_trans();
    }

    ++$current;
    $trans = $TRANS{$KEYS[$current]};
    ($date, $check, $desc, $debit, $credit, $cat, $com, $cleared, $junk) = 
    	    split(/\t/, $trans);
    while ( $cleared eq "x" ) {
        ++$current;
        $trans = $TRANS{$KEYS[$current]};
        ($date, $check, $desc, $debit, $credit, $cat, $com, $cleared, $junk) = 
    	        split(/\t/, $trans);
    }

    if ( $trans ) {
        return "$KEYS[$current]\t$trans";
    } else {
        return "none";
    }
}


# select transaction -- primes a transaction for future clearing
sub select_trans {
    # in: key
    # out: keyed_transaction

    local($key) = @_;
    $sorted_keys = 0;
    $calced = 0;

    $trans = $TRANS{$key};
    ($date, $check, $desc, $debit, $credit, $cat, $com, $cleared, $total) = 
    	    split(/\t/, $trans);

    $cleared = "*";

    $TRANS{$key} = 
	  "$date\t$check\t$desc\t$debit\t$credit\t$cat\t$com\t$cleared\t$total";

    print DEBUG "selected:  $key to be cleared\n" if $debug;

    return "$key\t$TRANS{$key}";
}


# select transaction -- primes a transaction for future clearing
sub unselect_trans {
    # in: key
    # out: keyed_transaction

    local($key) = @_;
    $sorted_keys = 0;
    $calced = 0;

    $trans = $TRANS{$key};
    ($date, $check, $desc, $debit, $credit, $cat, $com, $cleared, $total) = 
    	    split(/\t/, $trans);

    $cleared = "";

    $TRANS{$key} = 
	  "$date\t$check\t$desc\t$debit\t$credit\t$cat\t$com\t$cleared\t$total";

    print DEBUG "unselected:  $key will not be cleared\n" if $debug;

    return "$key\t$TRANS{$key}";
}


# clear all selected transactions
sub clear_trans {
    if ($calced == 0) {
	&calc_trans();
    }

    if ($sorted_keys == 0) {
	&sort_keys();
    }

    foreach $key (@KEYS) {
        $trans = $TRANS{$key};
        ($date, $check, $desc, $debit, $credit, $cat, $com, $cleared, $total) = 
    	        split(/\t/, $trans);

	if ( $cleared eq "*" ) {
            $cleared = "x";

            $TRANS{$key} = 
	          "$date\t$check\t$desc\t$debit\t$credit\t$cat\t$com\t$cleared\t$total";
        }
    }
}


# return the cleared balance (this should be the last statement ending bal)
sub get_cleared_bal {
    return sprintf("%.2f", $BALS{"Xamount"});
}


# initialize the transactions data structure
sub init_trans {
    # out: result

    $sorted_keys = 0;
    $calced = 0;
    @KEYS = ();
    $current_file = "";

    return "ok";
}


# make a new account
sub make_acct {
    # in: acct-name acct-desc acct-type
    # out: result
    
    local($name, $desc) = split(/ /, @_[0], 2);
    local($pos, $short_name);

    print DEBUG "Make account $name - $desc\n" if $debug;
    # print "Make account $name - $desc\n";

    print DEBUG "Making cbb account\n" if $debug;

    open(SAVE, ">$name.cbb.new");
    close(SAVE);
    unlink("$name.cbb.bak");
    rename("$name.cbb", "$name.cbb.bak");
    rename("$name.cbb.new", "$name.cbb");
    
    $current_file = "$name.cbb";

    # Assume we have category already open ... :| ??? :(

    # strip leading path from $name
    &insert_cat("[".&file_basename($name)."]\t$desc\t");

    # save the categories file before it gets toasted
    &save_cats(&file_dirname($name) . "/categories");

    return "ok";
}


# determine the file type and call the correct load routine
sub load_trans {
    # in: file base
    # out: result

    local($file) = @_;
    local($ext) = &file_extension($file);

    # print "$ext\n";
    # print &file_root($file) . "\n";

    print DEBUG "file extension is: $ext\n" if $debug;

    return &load_cbb_trans($file);
}


# load the data from a cbb file
sub load_cbb_trans {
    # in: file name (including .cbb extension)
    # out: result

    local($file) = @_;
    $sorted_keys = 0;
    $calced = 0;

    print DEBUG "Loading the cbb format file: $file\n" if $debug;

    open(LOAD, "<$file") || return "error";

    dbmclose(TRANS);    # close just in case the previous file was dbm
    %TRANS = ();	# clear out any transactions from the previous file
    $file_version = "";

    while ( <LOAD> ) {
	if ( m/^#/ ) {
	    # toss the comment (but first check for any goodies.)
	    if ( m/version/i ) {
		($junk, $junk, $junk, $file_version) = split;
		print DEBUG "Data file version = $file_version\n" if $debug;
	    }
	} else {
            chop;
	    if ( ! m/\t/ ) {
		s/:/\t/g;
		$_ = &fix_splits($_);
	    }
            &create_trans($_);
	}
    }

    close(LOAD);

    $current_file = $file;

    return "ok";
}


sub fix_splits {
    # in: transaction with old two field per record splits
    # out: transaction with new three field per record splits

    local($line) = @_;
    local($i);

    ($date, $check, $desc, $debit, $credit, $cat, $com, $cleared, $total) =
	split(/\t/, $line);

    if ( $cat =~ m/\|/ ) {
        @cats = split(/\|/, $cat);

        $i = 0;
        $max = ($#cats - 1) / 2;
        $newcat = "|";

        while ( $i < $max ) {
    	    $newcat .= $cats[$i * 2 + 1] . "||" . 
		       $cats[$i * 2 + 2] . "|";
	    $i++;
        }
    } else {
	$newcat = $cat;
    }

    return "$date\t$check\t$desc\t$debit\t$credit\t$newcat\t$com\t$cleared\t$total";
}


# load the data from a dbm file
sub load_dbm_trans {
    # in: file base name
    # out: result

    local($file) = @_;
    local($temp);

    print DEBUG "Loading the dbm format file: $file\n" if $debug;

    if ( -e "$file" ) {
	$current_file = $file;
	$sorted_keys = 0;
	$calced = 0;

	dbmclose(TRANS);
	dbmopen(%TRANS, &file_root($file), 0666) || return "error";
	
	# test to see if this file is <tab> delimited
	&sort_keys();
	# never ever call calc_trans() at this point (or call something that
	# calls it
	if ( defined($TRANS{$KEYS[0]}) && ! ($TRANS{$KEYS[0]} =~ m/\t/) ) {
	    print DEBUG "'$TRANS{$KEYS[0]}' = old version of CBB dbm file\n"
		if $debug;
	    return "error - old version of CBB dbm file";
	} else {
	    print DEBUG "valid txn: '$TRANS{$KEY[0]}'\n" if $debug;
        }

	return "ok";
    } else {
	return "error";
    }
}


# save all the precious data to a file
sub save_trans {
    # in: file name (including .cbb extension)
    # out: result

    local($file) = @_;
    local($auto_save_file);

    print DEBUG "Saving the cbb format file: $file\n" if $debug;

    if ($calced == 0) {
	&calc_trans();
    }

    if ($sorted_keys == 0) {
	&sort_keys();
    }

    open(SAVE, ">$file.new") || return "error";

    # Print some header stuff
    print (SAVE "# CBB Data File -- $file\n");
    print (SAVE "#\n");
    print (SAVE "# Version = $version_num\n");
    print (SAVE "# Saved on (US Date Fmt) " . &nice_date("1") . " ");
    print (SAVE "by $user_name\n");
    print (SAVE "#\n");
    print (SAVE "# date  check  desc  debit  credit  cat  com  cleared  total\n");
    print (SAVE "# ----------------------------------------------------------\n");

    foreach $key (@KEYS) {
        print (SAVE "$TRANS{$key}\n");
    }

    close(SAVE);

    unlink("$file.bak");
    rename("$file", "$file.bak");
    rename("$file.new", "$file");

    $auto_save_file = &file_dirname($file) . "#" . &file_basename($file) . "#";
    print DEBUG "auto_save_file = $auto_save_file\n" if $debug;
    if ( -e $auto_save_file ) {
	unlink("$auto_save_file");
	unlink("$auto_save_file.bak");
    }

    return "ok";
}


# ----------------------------------------------------------------------------
# $Log: engine.pl,v $
# Revision 1.32  1996/02/11  20:23:58  curt
# Removed support for dbm account files.
#
# Revision 1.31  1996/02/05  01:49:02  curt
# Changed several instances of cbb to CBB.
#
# Revision 1.30  1996/01/21  02:18:48  curt
# Modified Files:  categories.pl common.pl engine.pl export.pl file.pl import.pl
#                  log.pl memorized.pl reports.pl wrapper.pl
#   Updated copyright date.
#
# Revision 1.29  1995/08/18  11:34:07  curt
# Added support for the international date format.
#
# Revision 1.28  1995/08/03  22:58:02  curt
# Added support for long check #'s.
#
# Revision 1.27  1995/07/21  03:37:26  curt
# Fixed a really stupid bug with transfer transaction which caused them to fail.
#
# Revision 1.26  1995/07/20  11:18:55  curt
# Spelling fixes ...
#
# Revision 1.25  1995/07/20  03:21:17  curt
# Preformat transaction list entries, so tcl doesn't have to waste time
# doing it.
#
# Revision 1.24  1995/07/20  02:51:43  curt
# Mods to handle new split comment.
#
# Revision 1.23  1995/07/08  01:49:39  curt
# Change delimiter from ":" to "\t"
#
# Revision 1.22  1995/06/06  01:34:52  curt
# Modified Files:  categories.tk cbb engine.pl file.tk
#   Added a Yes/No/Cancel dialog box, and used it in places
#   Delete temporary autosave file after user requests a real save.
#
# Revision 1.21  1995/06/05  11:41:41  curt
# Modified Files:  Todo cbb engine.pl file.tk log.pl wrapper.pl
#   Fixed transfers between files.
#   Reworked transaction logging.
#   Added an auto-save feature.
#   Reworked .cbbrc saving and loading
#   Use '.dir' now instead of '.pag'
#
# Revision 1.20  1995/06/04  13:36:10  curt
# Added support for dual file formats (text and dbm).
# This involved modifications to make_acct(), load_trans(), save_trans().
#
# Revision 1.19  1995/04/24  01:57:22  curt
# Split of menu setup functions into a separate file:  menu.tk
# Started rearranging structure to support multiple file formats.
#
# Revision 1.18  1995/02/23  05:08:45  curt
# Removed dependencies on tclX.
#
# Revision 1.17  1994/11/07  23:14:08  clolson
# Working on interactive category viewing/editing
#
# Revision 1.16  1994/11/06  14:47:17  curt
# `19' prepended to transaction date when it is created if it is a 2-digit
# date.
#
# Revision 1.15  1994/11/04  19:32:58  clolson
# Added Transfers between Accounts.
#
# Revision 1.14  1994/11/02  12:58:39  curt
# Calculate the statement starting balance for the Balance operation.
#
# Revision 1.13  1994/11/01  22:39:04  clolson
# Added log.pl --> implemented logging.
# Worked on install script.
#
# Revision 1.12  1994/11/01  15:30:54  curt
# small changes.
#
# Revision 1.11  1994/10/31  23:14:50  clolson
# Started tracking towards cbbsh (the text only version).  Chris Browne has
# some good ideas, so I am trying to incorporate them, and make my stuff
# compatible with his.
#
# Revision 1.10  1994/10/31  15:01:41  curt
# engine.pl --> split into several pieces.
#
# Revision 1.9  1994/10/25  03:41:23  curt
# fixed pad function
#
# Revision 1.8  1994/10/24  21:30:31  curt
# Improved pad function - Thanks to Chris Browne.
#
# Revision 1.7  1994/10/19  13:27:26  curt
# Added export .qif files.
#
# Revision 1.6  1994/10/17  13:26:21  curt
# Added all-trans function.
#
# Revision 1.5  1994/10/14  17:05:18  clolson
# Miscellaneous cleanups in preparation for releasing version 0.40a
#
# Revision 1.4  1994/10/13  23:57:14  curt
# Finished memorized transactions.
#
# Revision 1.3  1994/10/13  22:08:56  clolson
# Worked on adding memorized transactions.
#
# Revision 1.2  1994/10/13  15:54:46  curt
# Turned debugging off by default.
#
# Revision 1.1  1994/10/11  15:05:01  curt
# Official name is now cbb (for now)
#
# Revision 1.22  1994/10/04  15:40:33  curt
# Categories saved when data is saved.
#
# Revision 1.21  1994/10/03  02:52:07  curt
# *.dat --> *.cbb
#
# Revision 1.20  1994/09/30  19:49:34  clolson
# Working on split category completion.
#
# Revision 1.19  1994/09/29  23:16:02  clolson
# Started work on categories
#
# Revision 1.18  1994/09/28  13:44:26  curt
# Restructured things so that data files look like file.dat, file.bat, file.cat
# Loading and saving are done by basename, i.e. file, data files from previous
# versions will have to be renamed to something.dat
#
# Also fixed a small bug where default tab bindings weren't being set at
# startup.
#
# Revision 1.17  1994/09/20  14:12:45  clolson
# Subjected code to gnu public license :)
#
# Revision 1.16  1994/09/14  03:49:25  clolson
# Worked on selecting and clearing transactions
#
# Revision 1.15  1994/09/12  15:19:45  curt
# Fixed first_uncleared_trans() and next_uncleared_trans() to really work
# right.
#
# Revision 1.14  1994/09/09  20:25:32  clolson
# Started work on first_uncleared_trans() and next_uncleared_trans()
#
# Revision 1.13  1994/09/07  23:07:48  clolson
# Minor modification to import_qif ... to toss blank lines
#
# Revision 1.12  1994/09/06  03:02:15  curt
# Removed data file '*.data' dependency
#
# Revision 1.11  1994/09/03  02:47:05  clolson
# Working on updating entries
#
# Revision 1.10  1994/09/02  03:40:28  curt
# Updated debugging messages.
# Removed print statement.
#
# Revision 1.9  1994/08/29  03:53:37  curt
# Added header to listbox (needs tweaking)
# Category now says -Splits- if there are splits.
#
# Revision 1.8  1994/08/29  00:16:23  curt
# Added calc_trans()
#
# Revision 1.7  1994/08/26  20:33:46  clolson
# Added @KEYS, $sorted_keys, and $current (documented in code)
# Added first_trans() and next_trans()
#
# Revision 1.6  1994/08/25  18:47:34  clolson
# Added a init_trans command.
#
# Revision 1.5  1994/08/25  04:17:14  curt
# Finished first stab at import_qif().
# Also finished load_dbm_trans() and save_dbm_trans().
#
# Revision 1.4  1994/08/25  02:08:58  clolson
# More work on .qif imports
#
# Revision 1.3  1994/08/24  04:09:11  curt
# Started working on routine to import qif (Quicken export) files.  But it is
# getting late so I must go to bed.
#
# Revision 1.2  1994/08/24  02:38:07  curt
# Added create, update, delete, and print transaction functions.
#
# Revision 1.1  1994/08/23  03:59:47  curt
# Initial stub of a transaction handling perl script.
#
