#include <Integer.h>
#include "common.h"
#include "rabin.h"
#include "quick_mod.h"
extern "C" {
  #include <sys/time.h>
}

typedef unsigned char BYTE;

BYTE *prepare_mod ( int *bigprep, int bignumbytes, int modulus )
{
  int i;
  int x = 1;
  int value = 1;
  BYTE *residues = new BYTE[modulus];
  if ( !residues )
    return NULL;
  for ( i = 0; i < bignumbytes; i++ ) {
    bigprep[i] = value;
    value <<= 8;
    value %= modulus;
  }
  for ( i = 0; i < modulus; i ++ )
    residues[i] = 0;
  for ( i = 0; i < modulus; i ++ )
    residues[(i*i)%modulus] = 1;
  return residues;
}

int quick_mod ( BYTE *bignum, int *bigprep, int bignumbytes, int modulus )
{
  int i;
  int result = 0;
  for ( i = 0; i < bignumbytes; i ++ )
    /* Shouldn't overflow for #'s not too long.  :-) */
    result += bigprep[i] * bignum[i];
  return result % modulus;
}

int check_sig ( BYTE *n, BYTE *m, BYTE *c, int modulus,
	       int *modprep, BYTE *residueprep, int bignumbytes )
{
  int np, mp, cp;
  np = quick_mod ( n, modprep, bignumbytes, modulus );
  mp = quick_mod ( m, modprep, bignumbytes, modulus );
  cp = quick_mod ( c, modprep, bignumbytes, modulus );
  return residueprep[(mp+np*cp)%modulus];
}

int prepare_for_check_sig ( Integer a, BYTE *bignum, int bignumbytes )
{
  int i, j, k, x;
  for ( i = j = 0; i < bignumbytes; i ++ ) {
    x = 0;
    for ( k = 0; k < 8; j++, k++ ) {
      int bit = testbit(a, j) ? 1 : 0;
      x += bit << k;
    }
    bignum[i] = x;
  }
}

#define BYTES (((SIZE_OF_KEYS*2)+31)/8)
#define MAXTRIES 100

int   modprep[MAXTRIES][BYTES];
BYTE *resprep[MAXTRIES];
int   num_tries = 0;
int   moduli[MAXTRIES];

void init_quick_sig ( int tries )
{
  int i;
  if ( tries > MAXTRIES ) {
    cout << "Internal try limit exceeded!\n";
    return;
  }
  if ( num_tries ) {
    for ( i = 0; i < num_tries; i ++ )
      delete resprep[i];
  }
  num_tries = tries;
  for ( i = 0; i < tries; i ++ ) {
    int modulus = generate_prime(TEMP_PRIME_SIZE).as_long();
    moduli[i] = modulus;
    resprep[i] = prepare_mod ( modprep[i], BYTES, modulus );
    if ( !resprep[i] ) {
      cout << "Allocated failed.\n";
      exit ( -1 );
    }
  }
}

int check_sig_quick ( rabin_public_key pubkey, signed_message m,
		     struct timeval *thing )
{
/*
  Integer a;
  BYTE aa[BYTES];
  a = 1;
  prepare_for_check_sig ( a, aa, BYTES );
  cout << "quick_mod: " << quick_mod ( aa, modprep[0], BYTES, moduli[0] ) << "\n";
  cout << "      mod: " << (a) % (moduli[0]) << "\n";
  a = 256;
  prepare_for_check_sig ( a, aa, BYTES );
  cout << "quick_mod: " << quick_mod ( aa, modprep[0], BYTES, moduli[0] ) << "\n";
  cout << "      mod: " << (a) % (moduli[0]) << "\n";
  a = 65536;
  prepare_for_check_sig ( a, aa, BYTES );
  cout << "quick_mod: " << quick_mod ( aa, modprep[0], BYTES, moduli[0] ) << "\n";
  cout << "      mod: " << (a) % (moduli[0]) << "\n";
  a = 1000000000L;
  prepare_for_check_sig ( a, aa, BYTES );
  cout << "quick_mod: " << quick_mod ( aa, modprep[0], BYTES, moduli[0] ) << "\n";
  cout << "      mod: " << (a) % (moduli[0]) << "\n";
  a *= 1234567890L;
  a += 75847334L;
  prepare_for_check_sig ( a, aa, BYTES );
  cout << "quick_mod: " << quick_mod ( aa, modprep[0], BYTES, moduli[0] ) << "\n";
  cout << "      mod: " << (a) % (moduli[0]) << "\n";
  a *= 8483247L;
  a += 349234L;
  prepare_for_check_sig ( a, aa, BYTES );
  cout << "quick_mod: " << quick_mod ( aa, modprep[0], BYTES, moduli[0] ) << "\n";
  cout << "      mod: " << (a) % (moduli[0]) << "\n";
*/
  int  result;
  BYTE nn[BYTES], mm[BYTES], cc[BYTES];
  prepare_for_check_sig ( pubkey.n, nn, BYTES );
  prepare_for_check_sig ( m.msg,    mm, BYTES );
  prepare_for_check_sig ( m.sig,    cc, BYTES );
  gettimeofday(thing, NULL);
  for ( int i = 0; i < num_tries; i ++ ) {
/*
    cout << "Modulus: " << moduli[i] << "\n";
    cout << "resprep: "
      << (char)(resprep[i][0]+'0') << (char)(resprep[i][1]+'0') << (char)(resprep[i][2]+'0') << (char)(resprep[i][3]+'0')
	<< (char)(resprep[i][4]+'0') << (char)(resprep[i][5]+'0') << (char)(resprep[i][6]+'0') << (char)(resprep[i][7]+'0')
	  << "\n";
    cout << "Mine: " << quick_mod ( nn, modprep[i], BYTES, moduli[i] );
    cout << " theirs: " << (pubkey.n) % moduli[i];
    cout << " mod: " << moduli[i] << "\n";
    cout << "Theirs: " << (pubkey.n) % moduli[i] << " "
      << (m.msg) % moduli[i] << " "
	<< (m.sig) % moduli[i] << "\n";
*/
    result = check_sig ( nn, mm, cc, moduli[i], modprep[i], resprep[i],
			BYTES );
    if ( !result )
      return 0;
  }
  return 1;
}
