use warnings;
use strict;

=head1 NAME

Doxory::Action::Login

=cut

package MyWebLog::Action::Login;
use base qw/Jifty::Action/;

=head2 arguments

Return the email and password form fields

=cut

sub arguments {
    return (
        {
            email => {
                label          => _('Email'),
                mandatory      => 1,
                ajax_validates => 1,
            },

            password => {
                type      => 'password',
                label     => _('Password'),
                mandatory => 1
            },
            remember => {
                type  => 'checkbox',
                label => _('Save login'),
                hints =>
                  _('Check this box and Doxory will just log you in next time'),
                default => 0,
            }
        }
    );

}

=head2 validate_email ADDRESS

Makes sure that the email submitted is a legal email address and that there's a user in the database with it.


=cut

sub validate_email {
    my $self  = shift;
    my $email = shift;
    my $LoginUser   = "MyWebLog::Model::User";
    my $CurrentUser = "MyWebLog::CurrentUser";

    # Let Jifty::Action and the mandatory flag take care of non-existant values
    return if not defined $email or not length $email;

    unless ( $email =~ /\S\@\S/ ) {
        return $self->validation_error(
            email => _("That doesn't look like an email address.") );
    }

    my $u = $LoginUser->new( current_user => $CurrentUser->superuser );
    $u->load_by_cols( email => $email );
    return $self->validation_error(
        email => _('No account has that email address.') )
      unless ( $u->id );

    return $self->validation_ok('email');
}

=head2 take_action

Actually check the user's password. If it's right, log them in.
Otherwise, throw an error.


=cut

sub take_action {
    my $self = shift;

    my $user = MyWebLog::CurrentUser->new( email => $self->argument_value('email') );

    unless ( $user->id
        && $user->password_is( $self->argument_value('password') ) )
    {
        $self->result->error(
 _('You may have mistyped your email address or password. Give it another shot?')
        );
        return;
    }

    unless ( $user->user_object->email_confirmed ) {
        $self->result->error(_("You haven't confirmed your account yet."));
        return;
    }

    # Set up our login message
    $self->result->message( _("Welcome back, ") . $user->user_object->name . "." );

    # Actually do the signin thing.
    Jifty->web->current_user($user);
    Jifty->web->session->expires(
        $self->argument_value('remember') ? '+1y' : undef );
    Jifty->web->session->set_cookie;

    return 1;
}

1;
