#!/usr/bin/perl
#
# Periodically scan the list of available wireless networks and adjust
# the ESSID in the card according to a preferred-SSID list.
#
# Nickolai Zeldovich <kolya@MIT.EDU>
#
# Version 1.1:
#    Added support for wireless cards that don't support `iwlist scan`,
#    suggested and implemented by Ramesh Chandra.
#
# $Id: wifiassocd.pl,v 1.11 2004/04/12 21:46:07 kolya Exp $

use utf8;
use strict;

my $CONFIG = "/etc/wifiassocd.conf";
my $DEVICE;
my $PERIOD;
my @PREFER;
my $DEBUG = 0;

foreach my $arg (@ARGV) {
    if ($arg eq '-d') {
	$DEBUG = 1;
    } else {
	print	"Unknown argument: $arg\n",
		"\n",
		"Usage: wifiassocd.pl [-d]\n",
		"  -d    Enable debug mode\n";
	exit 1;
    }
}

exit if !$DEBUG && fork();

my $lastssid;
my $down = 1;
while (1) {
    read_config();

    if (!$down && is_associated()) {
	print "Still associated, sleeping for $PERIOD seconds\n" if $DEBUG;
	sleep $PERIOD;
	next;
    }

    my $newssid;
    my @availssid = scan();
    print "SSID list: ", join(', ', @availssid), "\n" if $DEBUG;

    foreach my $ssid (@PREFER) {
	if (defined @availssid) {
	    if (grep {/^$ssid$/} @availssid) {
		print "Found preferred SSID $ssid\n" if $DEBUG;
		$newssid = $ssid;
		last;
	    }
	} else {
	    print "No scan results -- probing SSID $ssid\n" if $DEBUG;
	    set_ssid($DEVICE, $ssid);
	    if (is_associated()) {
		print "SSID $ssid seems to associate\n" if $DEBUG;
		$newssid = $ssid;
		last;
	    }
	}

	print "Preferred SSID $ssid not found in scan\n" if $DEBUG;
    }

    if (!(defined $newssid)) {
	print "No preferred SSID found, trying to guess\n" if $DEBUG;

	my $ssid_guess;
	if (defined $availssid[0]) {
	    $ssid_guess = $availssid[0];
	    print "Guessing SSID from scan list: $ssid_guess\n" if $DEBUG;
	} else {
	    $ssid_guess = "any";
	    print "No scan list available, using 'essid any'\n" if $DEBUG;
	}
	set_ssid($DEVICE, $ssid_guess);
	$newssid = get_assoc_ssid();
	print "Guessed <$ssid_guess>, now connected to <$newssid>\n" if $DEBUG;

	if ((defined $newssid) && !(grep {/^$newssid$/} @PREFER)) {
	    append_ssid($newssid);
	}
    }

    if (defined $newssid) {
	if ($newssid ne $lastssid) {
	    print "Changing association from old <$lastssid> to ",
		  "new <$newssid>\n" if $DEBUG;
	    $lastssid = $newssid;
	    change_ssid($DEVICE, $newssid);
	} else {
	    print "Still associated to $newssid\n" if $DEBUG;
	}
	$down = 0;
    } else {
	print "No available SSIDs, down'ing the interface\n" if $DEBUG;
	xsystem("ifdown $DEVICE");
	$down = 1;
    }

    print "Done with scan, sleeping for $PERIOD seconds\n\n" if $DEBUG;
    sleep $PERIOD;
}

sub is_associated {
    my $associated = get_assoc_ap();

    if ($associated ne 'FF:FF:FF:FF:FF:FF' &&
	$associated ne '00:00:00:00:00:00' &&
	$associated ne '44:44:44:44:44:44' &&
	$associated ne 'unknown') {
	return 1;
    } else {
	return 0;
    }
}

sub get_assoc_ap {
    my $out = `iwconfig $DEVICE`;
    return $1 if $out =~ /Access Point: (..:..:..:..:..:..)/;
    print "Error: unable to determine associated access point for $DEVICE\n"
	if $DEBUG;
    return "unknown";
}

sub get_assoc_ssid {
    my $out = `iwconfig $DEVICE`;
    return $1 if $out =~ /ESSID:"([^"]*)"/;
    print "Device doesn't seem to be associated\n" if $DEBUG;
    return undef;
}

sub scan {
    my @ssid = ();
    xsystem("ifconfig $DEVICE up");
    open(IWSCAN, "iwlist $DEVICE scan |");
    while (<IWSCAN>) {
	return () if /Operation\s+not\s+supported/;
	push @ssid, $1 if /ESSID:"([^"]*)"$/;
    }
    return @ssid;
}

sub read_config {
    $DEVICE = '';
    $PERIOD = '';
    @PREFER = ();

    open(F, "$CONFIG");
    while (<F>) {
	s/#.*$//;
	s/\s*$//;
	$DEVICE = $1 if /device (.*)/;
	$PERIOD = $1 if /period (.*)/;
	push @PREFER, &url_unquote($1) if /ssid \"(.*)\"/;
    }
    close(F);

    if ($DEBUG) {
	print	"Configuration:\n",
		"  Device: $DEVICE\n",
		"  Period: $PERIOD\n",
		"  SSIDs: ", join(', ', @PREFER), "\n";
    }
}

sub append_ssid {
    my ($ssid) = @_;
    print "Appending SSID $ssid to config file\n" if $DEBUG;
    open(F, ">>$CONFIG");
    print F
	"\n",
	"# ", `date`,
	"# Automatically added by wifiassocd\n",
	"ssid \"", &url_quote($ssid), "\"\n";
    close(F);
}

sub set_ssid {
    my ($dev, $ssid) = @_;
    xsystem("iwconfig $dev essid '$ssid' > /dev/null 2>&1");
    sleep 3;
}

sub change_ssid {
    my ($dev, $ssid) = @_;

    xsystem("ifdown $dev >/dev/null 2>&1");

    # Rewrite /etc/sysconfig/network-scripts/ifcfg-$dev to have ESSID=$ssid
    my $ifcfg = "/etc/sysconfig/network-scripts/ifcfg-$dev";

    open(I, "$ifcfg");
    open(O, ">$ifcfg.new");

    my $wrotessid = 0;
    while (<I>) {
	if (/^ESSID=/) {
	    print O "ESSID=\"$ssid\"\n";
	    $wrotessid = 1;
	} else {
	    print O;
	}
    }
    if (!$wrotessid) {
	print O "ESSID=\"$ssid\"\n";
    }
    close(O);
    xsystem("mv $ifcfg.new $ifcfg");

    # Redhat sucks
    xsystem("find /etc/sysconfig/networking -name ifcfg-$dev | xargs rm");

    xsystem("ifup $dev >/dev/null 2>&1");
}

sub xsystem {
    my (@args) = @_;

    print "Running command: ", join(' ', @args), "\n" if $DEBUG;
    system(@args);
}

sub url_quote {
    my $s = shift;
                                                                                
    utf8::upgrade($s) unless $^V and $^V lt v5.8.0;
                                                                                
    $s =~ s/([^a-zA-Z0-9_.\*~-])/sprintf("%%%02X",ord($1))/eg;
                                                                                
    return $s;
}

sub url_unquote {
    my $out = shift;
                                                                                
    $out =~ s/\+/ /g;
    $out =~ s/%([[:xdigit:]][[:xdigit:]])/ chr hex($1)/ge;
 
    utf8::downgrade($out) unless $^V and $^V lt v5.8.0;
                                                                                
    return $out;
}
